/*
 * compare - compare file hierarchies
 *
 * compare [flags] [[user@]machine[!program]:]directory
 *				[[user@]machine[!program]:]directory ...
 *
 * flags (old forms still accepted)
 *	-update (old form: -u)
 *	-follow (old form: -h)
 *	-sgid-dirs (old form: -g)
 *	-mtimes (old form: -t)
 *	-no-owners (old form: -o)
 *	-encode (old form: -x)
 *	-no-sparse
 *	-trace
 *	-prune <name>
 *	-prunewild <pattern>
 *	-prunex
 *	-md5
 *	-gzip <gzip-arg>
 *	-no-delete
 *	-rsh <program>
 *	-dir <dir>
 *	-mask <mask>
 *	-modemod <and> <or>
 *	-tick <seconds>
 *	-links
 *	-ignore-nonexistent (also -inx) (applies to next directory)
 *	-readonly (also -ro)
 *	-check
 *
 *	-R
 *	-accept <port-number>[:<dotted-quad>[/<number|dotted-quad>]]
 *	-connect <dotted-quad> <port-number>
 *	-force <dir>
 *	-bg
 *	-limit
 *
 * Compare compares directories, possibly on different machines, with
 *  one another.
 *
 * Each (non-option) argument specifies a directory.  If no :s occur in
 *  the argument, or if a slash appears before the first : or !, the
 *  directory is on the local machine.  Otherwise, the portion before
 *  the colon specifies a machine name with optional user and program
 *  specifications.  By default, compare uses rsh(1), which must be in
 *  the path, to run compare on the remote machine; see the -rsh option
 *  for a way to change this.  If a username is given with an @ before
 *  the machine name, it will be used with rsh's -l option to specify
 *  the remote username.  If a program name is specified after the
 *  machine name with a !, that will be the name used to run compare on
 *  the remote machine.  The default for the username is whatever rsh
 *  feels like using (normally the local username); the default for the
 *  program name is the argv[0] value for the master run.
 *
 * If the machine name is an empty string (ie, the argument begins with
 *  a colon), the argument is taken to be of the form
 *	:keyword:info
 *  to specify an alternative way of contacting the remote.  Currently,
 *  the only defined ways are
 *	:connect:dotted-quad:port-number:directory
 *  and
 *	:accept:port-number:directory
 *  These cause compare to do a connect or a listen/accept with the
 *  given port number (and address, for "connect") to establish the
 *  connection to the remote compare process.  This is useful when you
 *  have shell access on two machines but can't (or don't want to)
 *  allow rsh access between them.  See the -R, -accept, and -connect
 *  options for more.  Note that certain options must normally agree
 *  between the master and the remotes, either given to both or given
 *  to neither.  These options are -mtimes and -encode.  In addition,
 *  -gzip must be given to the remote if it's given to the master,
 *  though the converse is not true (if the master is not given -gzip,
 *  it will not matter whether the remotes are or not).  Other flags
 *  (notably -follow, -no-owners, and -sgid-dirs) may usefully vary
 *  between the master and the remotes; some others (-update and
 *  -mtimes, for example) affect only the master and hence have no
 *  effect when given to a remote.
 *
 * -dir takes its following argument as a directory to compare.  It
 *  exists to allow specifying a directory that begins, or may begin,
 *  with a -.
 *
 * -update means to update: the first-named directory is treated as a
 *  master copy and all others are made identical to it, like a
 *  super-picky rdist (compare compares more things than rdist does).
 *
 * -follow means to follow symlinks instead of checking the links
 *  themselves for matching.
 *
 * -sgid-dirs says to ignore the set-group-ID bits on directories when
 *  comparing modes and to leave them alone when setting modes.  It's
 *  like a variant of "-mask 5777" that applies only to directories.
 *
 * -mtimes says that modification times are important and should be
 *  considered when comparing for differences and preserved when
 *  updating.
 *
 * -no-owners says that the owning UID and GID values for files are
 *  unimportant and should not be set or compared.
 *
 * -encode says that network communication should be passed through a
 *  simple binary-to-printable filter.  (This _should_ never be needed,
 *  but the world is far from ideal, and it does seem to help in some
 *  cases.)
 *
 * -no-sparse says that files should never be created sparse.  Without
 *  this, files compare creates will always be as sparse as possible.
 *
 * -trace says to trace data sent and received between the master
 *  process and the auxiliaries.  (If -encode is in effect, this traces
 *  the encoded form as well.)  Trace output is sent to stderr.
 *
 * -prune <name> says that anything called <name> (which should be a
 *  path relative to the directories being compared) should be ignored
 *  when found: neither it nor anything under it should be compared,
 *  updated, or otherwise touched.  compare behaves as if no such names
 *  existed anywhere, except that if a directory is copied wholesale
 *  because it was nonexistent during an update operation, -prune
 *  entries referring to things inside that directory will not limit
 *  what is copied.
 *
 * -prunewild <pattern> is just like -prune except that the argument is
 *  a globbing pattern rather than a simple name.
 *
 * -prunex says that that the sense of the tests used for pruning
 *  should be reversed: everything is pruned _except_ the things named
 *  with -prune or -prunewild arguments.
 *
 * -md5 says that when plain files contents' need to be compared,
 *  rather than comparing the full contents, simply compute an md5
 *  checksum of each file, and assume that matching checksums means
 *  matching contents.  (This is useful when running over a network
 *  link sufficiently slow that it takes longer to send the file than
 *  it does to compute its checksum, and you're willing to take the
 *  slight risk that different files will checksum the same.)
 *
 * -gzip says that when copying files from one place to another, the
 *  files are to be gzipped at the point of reading and gunzipped at
 *  the point of writing.  (The intention is to reduce bytes sent over
 *  the net; this is intended for use across slow links, where cpu
 *  cycles are cheaper than network bytes.)  The argument is passed to
 *  gzip; it is expected to be something like --best or --fast.
 *
 * -no-delete says that an update operation (see -update) should ignore
 *  files that exist on the remote but not the master.  (Normally, such
 *  files would be deleted.)
 *
 * -rsh <program> says to use the specified <program> instead of rsh
 *  for running remote programs.  This is not a global option; it
 *  affects directory specifications to the right of it (until another
 *  -rsh option), but not those to the left of it.  The <program> is
 *  run as either
 *	<program> machine-name command [args...]
 *  or
 *	<program> machine-name -l username command [args...]
 *  where machine-name, username, and command are as described above,
 *  and args is zero or more arguments to be passed to the remote
 *  executable's command line.
 *
 * -mask specifies what bits to pay attention to when comparing
 *  permissions.  Both objects' mode bits are ANDed with the -mask
 *  argument before comparison.  The argument is an octal number.  (The
 *  mode bits are not modified; if (for example) a file's contents
 *  differ, the mode bits printed will be the actual mode bits, not
 *  masked by the -mask argument.)
 *
 * -modemod specifies that when -update is in effect, all mode bits
 *  read from the master directory tree are to be modified before
 *  comparing, printing, or setting on the other directory trees.
 *  Everything is ANDed with the <and> argument and then ORed with the
 *  <or> argument.  (Only the 07777 bits in each argument are
 *  significant.)
 *
 * -tick gives an interval in seconds; every this many seconds, the
 *  current path being worked on will be printed.  (Ticks occurring
 *  before all remotes start are ignored.)  A tick can also be forced
 *  by explicitly sending a SIGALRM, either from the command line via
 *  kill(1) or programmatically with kill(2), whether or not -tick was
 *  given.  (The signal must be sent to the parent compare process,
 *  which almost always means the lowest PID.)  This output, like that
 *  generated by SIGINFO, goes to /dev/tty.  If /dev/tty can't be
 *  opened at startup, SIGINFO and SIGALRM output is discarded.
 *
 * -links says that link count is an attribute that should be compared.
 *  Since there is no simple way to find other links, -u is not capable
 *  of fixing link count differences - indeed, the other links may not
 *  be under the tree being compared, or may not even have names, and
 *  thus may not be fixable.
 *
 * -ignore-nonexistent (which can also be spelled -inx) specifies that
 *  the next directory argument is to be treated slightly differently:
 *  if the only reason to print a difference report is that an entry is
 *  nonexistent in one or more so-marked directories, then the report
 *  is not printed.  This option should not be used with -u; the result
 *  is "whatever the implementation gives you", not anything
 *  well-specified.
 *
 * -check is designed to verify that all directories match; if they do,
 *  exit status is 0, if not, 1.  No output is produced for
 *  differences, nor does compare progress past the first difference
 *  discovered.  -check is useful only on the master run; it is not
 *  passed automatically to remotes, and remotes ignore it if it's
 *  given on a run-by-hand remote.
 *
 * -readonly is used with -R; it specifies that the remote is to reject
 *  any commands which would modify anything.
 *
 * -R says that this compare process is a remote.  It is normally not
 *  needed unless you're using explicit rendezvous points instead of
 *  the usual rsh way of starting the remotes.
 *
 * -accept is useful only with -R.  It specifies that the process is to
 *  listen/accept on the given port number to establish the connection
 *  to the master, instead of assuming the connection is already
 *  present on stdin and stdout.  -R -accept is used when the master
 *  uses a :connect:-style remote specifier.  If the optional colon is
 *  given, the dotted-quad specifies the IP address the remote peer
 *  must be on (or the connection is dropped, a complaint is printed to
 *  stderr, and a new connection is accepted).  A mask may be given,
 *  either as a CIDR width or as a dotted-quad mask, in which case any
 *  address on that net is accepted.
 *
 * -connect is useful only with -R.  It specifies that the process is
 *  to connect to the given port number at the given address to
 *  establish the connection to the master, instead of assuming the
 *  connection is already present on stdin and stdout.  -R -connect is
 *  used when the master uses a :accept:-style remote specifier.
 *
 * -force is useful only with -R.  It specifies that the remote is to
 *  ignore the directory that is specified at protocol startup, using
 *  the -force argument instead.
 *
 * -bg is useful only with -R.  It specifies that the remote is to
 *  background itself as soon as any initial setup (such as listening
 *  for -accept) is completed.
 *
 * -limit is useful only with -R.  It specifies that the remote is to
 *  refuse to accept a master-specified path that begins with a slash
 *  or attempts to dot-dot its way up the directory tree.
 *
 * No provision is made for user names containing /s, @s, or :s,
 *  machine names containing /s, !s or :s, or program names containing
 *  :s.
 */

#include <md5.h>
#include <poll.h>
#include <stdio.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <strings.h>
#include <sys/dir.h>
#include <sys/uio.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/resource.h>

/* fixup for SunOS versions which have <poll.h> and  poll() but not these */
#ifndef POLLRDNORM
#define POLLRDNORM 0
#endif
#ifndef POLLRDBAND
#define POLLRDBAND 0
#endif
#ifndef POLLPRI
#define POLLPRI 0
#endif
#ifndef INFTIM
#define INFTIM (-1)
#endif
#if defined(sparc) && defined(sun) /*&& defined(unix)*/ /* SunOS */
#define lchmod(x,y) ((errno=EINVAL),-1)
#define lchown(x,y,z) ((errno=EINVAL),-1)
#define strtoul(s,e,b) ((unsigned long int)strtol(s,e,b))
#endif

#define CURRENT_VERSION 1

#ifdef NO_PROGNAME
const char *__progname;
int main(int, char **);
int main_(int, char **);
int main(int ac, char **av) { __progname = av[0]; return(main_(ac,av)); }
#define main main_
#endif

#ifdef DECLARE_ALARM
extern void alarm(unsigned int);
#endif

extern const char *__progname;

#define NEW(t) ((t *)malloc(sizeof(t)))
#define OLD(x) free((x))

/* code assumes WORKSIZE is a multiple of 512 */
#define WORKSIZE 10240
#define MAXREMOTES 64

#define COPYBLK_DATA 1
#define COPYBLK_HOLE 2

#ifdef NO_FIFO
#undef S_IFIFO
#endif

static int flag_debug;
static int flag_follow;
static int flag_sgiddir;
static int flag_noown;
static int flag_remote;
static int flag_mtimes;
static int flag_update;
static int flag_encode;
static int flag_trace;
static int flag_md5;
static int flag_gzip;
static int flag_nosparse;
static int flag_nodelete;
static int flag_accept;
static int flag_connect;
static int flag_links;
static int flag_nextinx;
static int flag_readonly;
static int flag_check;
static int flag_bg;
static int flag_limitslave;

static struct sockaddr_in conn_sin;
static unsigned long int peer_addr;
static unsigned long int peer_mask;
static int perm_mask;
static int perm_and = ~0;
static int perm_or = 0;
static int tick_seconds;
static int trace_ctl = -1;
static char *force_dir = 0;

static char *defprogram;

typedef struct mask MASK;
typedef struct entry ENTRY;
typedef struct entry_tconc ENTRY_TCONC;
typedef struct remote REMOTE;
typedef struct rstack RSTACK;
typedef struct prune PRUNE;
typedef struct mvlist MVLIST;

#define MASKLONGS ((MAXREMOTES+31)>>5)

struct mask {
  unsigned long int bits[MASKLONGS];
  } ;

#if MASKLONGS == 1
#define MASK_ZERO(m) ((m).bits[0] = 0)
#define MASK_SET(m,bit) ((m).bits[0] |= (1 << (bit)))
#define MASK_CLR(m,bit) ((m).bits[0] &= ~(1 << (bit)))
#define MASK_TST(m,bit) ((m).bits[0] & (1 << (bit)))
#define MASK_ISZERO(m) ((m).bits[0] == 0)
#define MASK_EQ(m1,m2) ((m1).bits[0] == (m2).bits[0])
#define MASK_ONEBIT(m) ((m).bits[0] && (((m).bits[0] & ((m).bits[0]-1)) == 0))
#else
static const MASK zmask; /* never changed */
#define MASK_ZERO(m) ((m) = zmask)
#define MASK_SET(m,bit) ((m).bits[(bit)>>5] |= (1 << ((bit)&31)))
#define MASK_CLR(m,bit) ((m).bits[(bit)>>5] &= ~(1 << ((bit)&31)))
#define MASK_TST(m,bit) ((m).bits[(bit)>>5] & (1 << ((bit)&31)))
#define MASK_ISZERO(m) mask_iszero(m)
#define MASK_EQ(m1,m2) mask_eq(m1,m2)
#define MASK_ONEBIT(m) mask_onebit(m)
#endif

struct mvlist {
  MVLIST *link;
  MASK m;
  unsigned long int v;
  } ;

struct prune {
  PRUNE *link;
  unsigned int type : 1;
#define PT_PATH 0
#define PT_WILD 1
  char *path;
  } ;

struct entry {
  ENTRY *link;
  char *name;
  int mode;
  int uid;
  int gid;
  long int mtime;
  int nlinks;
  dev_t rdev;
  unsigned long int size;
  char *linkto;
  MASK present;
  } ;

struct entry_tconc {
  ENTRY *list;
  ENTRY **tail;
  } ;

struct remote {
  REMOTE *link;
  char *fullarg;
  char *user;
  char *machine;
  char *program;
  char *directory;
  const char *rsh;
  int rfd;
  int wfd;
  unsigned char accepting : 1;
  unsigned char connecting : 1;
  unsigned char errored : 1;
  unsigned char ignnx : 1;
  FILE *rf;
  FILE *wf;
  int bit;
  ENTRY *dirlist;
  int worklen;
  char work[WORKSIZE];
  } ;

struct rstack {
  RSTACK * volatile link;
  ENTRY * volatile list;
  ENTRY * volatile cur;
  } ;

#define EACH_REM(var) var=remotes;var;var=var->link

static const char *currsh = "rsh";
static REMOTE *remotes;
static REMOTE *umaster;
static MASK active;
static const char *cur_dir;
static char *cur_ent;
static int cur_pruned;
static MASK cur_nx;
static int prunex;
static int needhdg;
static PRUNE *prunelist;
static char *gzip_arg;
static RSTACK * volatile rstack;

static char slavebase[WORKSIZE];
static char slavework[WORKSIZE];
static char slavetemp[WORKSIZE];

static char zeroblk[WORKSIZE];

static ENTRY *freeentries;

static __inline__ unsigned long int ulmin(unsigned long int, unsigned long int) __attribute__((__const__));
static __inline__ unsigned long int ulmin(unsigned long int a, unsigned long int b)
{
 return((a<b)?a:b);
}

static char *copyofstr(char *s)
{
 char *rv;

 rv = malloc(strlen(s)+1);
 strcpy(rv,s);
 return(rv);
}

static void panic(const char *fmt, ...)
{
 va_list ap;

 va_start(ap,fmt);
 fprintf(stderr,"%s: INTERNAL BUG: ",__progname);
 vfprintf(stderr,fmt,ap);
 fprintf(stderr,"\n");
 va_end(ap);
 abort();
}

static ENTRY *newentry(void)
{
 ENTRY *e;

 if (freeentries)
  { e = freeentries;
    freeentries = e->link;
  }
 else
  { e = NEW(ENTRY);
  }
 e->name = 0;
 e->linkto = 0;
 return(e);
}

static void freeentry(ENTRY *e)
{
 if (e->name) free(e->name);
 if (e->linkto) free(e->linkto);
 e->link = freeentries;
 freeentries = e;
}

static ENTRY *copyentry(ENTRY *e)
{
 ENTRY *new;

 new = newentry();
 *new = *e;
 if (e->name) new->name = copyofstr(e->name);
 if (e->linkto) new->linkto = copyofstr(e->linkto);
 return(new);
}

#if MASKLONGS > 1
static int mask_iszero(MASK m)
{
 int i;

 for (i=0;i<MASKLONGS;i++) if (m.bits[i]) return(0);
 return(1);
}
#endif

#if MASKLONGS > 1
static int mask_eq(MASK m1, MASK m2)
{
 int i;

 for (i=0;i<MASKLONGS;i++) if (m1.bits[i] != m2.bits[i]) return(0);
 return(1);
}
#endif

#if MASKLONGS > 1
static int mask_onebit(MASK m)
{
 int i;

 for (i=0;i<MASKLONGS;i++) if (m.bits[i]) break;
 if (i >= MASKLONGS) return(0);
 if (m.bits[i] & (m.bits[i]-1)) return(0);
 for (i++;i<MASKLONGS;i++) if (m.bits[i]) return(0);
 return(1);
}
#endif

static MASK mask_or(MASK a, MASK b)
{
#if MASKLONGS == 1
 a.bits[0] |= b.bits[0];
#else
 int i;
 for (i=0;i<MASKLONGS;i++) a.bits[i] |= b.bits[i];
#endif
 return(a);
}

#if MASKLONGS == 1
#define mask_setor(a,b) ((a)->bits[0] |= (b).bits[0])
#else
static void mask_setor(MASK *a, MASK b)
{
 int i;

 for (i=0;i<MASKLONGS;i++) a->bits[i] |= b.bits[i];
}
#endif

static MASK mask_and(MASK a, MASK b)
{
#if MASKLONGS == 1
 a.bits[0] &= b.bits[0];
#else
 int i;
 for (i=0;i<MASKLONGS;i++) a.bits[i] &= b.bits[i];
#endif
 return(a);
}

static MASK mask_andnot(MASK a, MASK b)
{
#if MASKLONGS == 1
 a.bits[0] &= ~b.bits[0];
#else
 int i;
 for (i=0;i<MASKLONGS;i++) a.bits[i] &= ~b.bits[i];
#endif
 return(a);
}

#if MASKLONGS == 1
#define mask_setandnot(a,b) ((a)->bits[0] &= ~(b).bits[0])
#else
static void mask_setandnot(MASK *a, MASK b)
{
 int i;

 for (i=0;i<MASKLONGS;i++) a->bits[i] &= ~b.bits[i];
}
#endif

static void bin2hex(unsigned char *bin, unsigned char *hex, int n)
{
 for (;n>0;n--)
  { *hex++ = 48 + ((*bin)>>4);
    *hex++ = 48 + ((*bin)&15);
    bin ++;
  }
}

static void hex2bin(unsigned char *hex, unsigned char *bin, int n)
{
 for (;n>0;n--)
  { *bin++ = ((hex[0] & 15) << 4) | (hex[1] & 15);
    hex += 2;
  }
}

static void trace_print(const char *tag, const char *tag2, void *bvp, int nb)
{
 unsigned char *bp;

 fprintf(stderr,"%s%s[%d] ",tag,tag2,nb);
 for (bp=bvp;nb>0;nb--,bp++)
  { switch (*bp)
     { case '\a': fprintf(stderr,"\\a"); break;
       case '\b': fprintf(stderr,"\\b"); break;
       case '\33':fprintf(stderr,"\\e"); break;
       case '\f': fprintf(stderr,"\\f"); break;
       case '\n': fprintf(stderr,"\\n"); break;
       case '\r': fprintf(stderr,"\\r"); break;
       case '\t': fprintf(stderr,"\\t"); break;
       case '\v': fprintf(stderr,"\\v"); break;
       default:
	  if ((*bp < 32) || ((*bp > 126) && (*bp <= 160)))
	   { if (bp[1] && (bp[1] >= '0') && (bp[1] <= '9'))
	      { fprintf(stderr,"\\%03o",*bp);
	      }
	     else
	      { fprintf(stderr,"\\%o",*bp);
	      }
	   }
	  else
	   { putc(*bp,stderr);
	   }
	  break;
     }
  }
 putc('\n',stderr);
}

static void run_tracer(int ctl, int death)
{
 typedef struct client CLIENT;
 struct client {
   CLIENT *link;
   int fd;
   char *lbuf;
   int lalloc;
   int lfill;
   int fdx;
   } ;
 CLIENT *clients;
 struct pollfd *fdv;
 int fdva;
 int fdvn;
 CLIENT *c;
 CLIENT **cp;

 static int addfd(int fd)
  { struct pollfd *f;
    if (fdvn >= fdva) fdv = realloc(fdv,(fdva=fdvn+1)*sizeof(*fdv));
    f = &fdv[fdvn];
    f->fd = fd;
    f->events = POLLIN | POLLRDNORM | POLLRDBAND | POLLPRI;
    return(fdvn++);
  }

 static void client_line(CLIENT *c, char *addbuf, int addlen)
  { struct iovec iov[3];
    int iox;
    static char nl = '\n';
    iox = 0;
    if (c->lfill > 0)
     { iov[iox].iov_base = c->lbuf;
       iov[iox].iov_len = c->lfill;
       iox ++;
     }
    if (addlen > 0)
     { iov[iox].iov_base = addbuf;
       iov[iox].iov_len = addlen;
       iox ++;
     }
    iov[iox].iov_base = &nl;
    iov[iox].iov_len = 1;
    iox ++;
    writev(2,&iov[0],iox);
    c->lfill = 0;
  }

 static void client_read(CLIENT *c)
  { char rbuf[512];
    int rn;
    char *nl;
    int rx;
    int l;
    rn = read(c->fd,&rbuf[0],sizeof(rbuf));
    if (rn < 0)
     { fprintf(stderr,"%s: tracer client read: %s\n",__progname,strerror(errno));
       exit(1);
     }
    if (rn == 0)
     { if (c->lfill) client_line(c,0,0);
       close(c->fd);
       c->fd = -1;
       return;
     }
    rx = 0;
    while ((nl = memchr(&rbuf[rx],'\n',rn-rx)))
     { client_line(c,&rbuf[rx],nl-&rbuf[rx]);
       c->lfill = 0;
       rx = (nl+1) - &rbuf[0];
     }
    l = rn - rx;
    if (l > 0)
     { if (c->lfill+l > c->lalloc) c->lbuf = realloc(c->lbuf,c->lalloc=c->lfill+l);
       bcopy(&rbuf[rx],c->lbuf+c->lfill,l);
       c->lfill += l;
     }
  }

 static void ctl_read(void)
  { struct msghdr mh;
    struct iovec iov;
#ifdef SCM_RIGHTS
    struct cmsghdr cmh;
    char cbuf[sizeof(struct cmsghdr)+sizeof(int)];
#endif
    int fd;
    char junk;
    CLIENT *c;
    mh.msg_name = 0;
    mh.msg_namelen = 0;
    iov.iov_base = &junk;
    iov.iov_len = 1;
    mh.msg_iov = &iov;
    mh.msg_iovlen = 1;
#ifdef SCM_RIGHTS
    mh.msg_control = &cbuf[0];
    mh.msg_controllen = sizeof(cbuf);
    mh.msg_flags = 0;
#else
    mh.msg_accrights = (void *)&fd;
    mh.msg_accrightslen = sizeof(int);
#endif
    if (recvmsg(ctl,&mh,0) < 0)
     { fprintf(stderr,"%s: can't receive trace fd: %s\n",__progname,strerror(errno));
       exit(1);
     }
#ifdef MSG_CTRUNC
    if (mh.msg_flags & MSG_CTRUNC)
     { fprintf(stderr,"%s: can't receive trace fd (control info truncated)\n",__progname);
       exit(1);
     }
#endif
#ifdef SCM_RIGHTS
    if (mh.msg_controllen < sizeof(cbuf))
     { fprintf(stderr,"%s: can't receive trace fd (control info unexpectedly short)\n",__progname);
       exit(1);
     }
    bcopy(&cbuf[0],&cmh,sizeof(cmh));
    if (cmh.cmsg_len != sizeof(cbuf))
     { fprintf(stderr,"%s: can't receive trace fd (control packet len wrong)\n",__progname);
       exit(1);
     }
    if (cmh.cmsg_level != SOL_SOCKET)
     { fprintf(stderr,"%s: can't receive trace fd (control info level wrong)\n",__progname);
       exit(1);
     }
    if (cmh.cmsg_type != SCM_RIGHTS)
     { fprintf(stderr,"%s: can't receive trace fd (control info type wrong)\n",__progname);
       exit(1);
     }
    bcopy(&cbuf[sizeof(cmh)],&fd,sizeof(int));
#else
    if (mh.msg_accrightslen != sizeof(int))
     { fprintf(stderr,"%s: can't receive trace fd (access rights unexpectedly short)\n",__progname);
       exit(1);
     }
#endif
    c = malloc(sizeof(CLIENT));
    c->fd = fd;
    c->lbuf = 0;
    c->lalloc = 0;
    c->lfill = 0;
    c->link = clients;
    clients = c;
  }

 clients = 0;
 fdv = 0;
 fdva = 0;
 while (1)
  { fdvn = 0;
    addfd(ctl);
    if (death >= 0) addfd(death);
    cp = &clients;
    while ((c = *cp))
     { if (c->fd >= 0)
	{ cp = &c->link;
	  c->fdx = addfd(c->fd);
	}
       else
	{ *cp = c->link;
	  free(c->lbuf);
	  free(c);
	}
     }
    if ((death < 0) && !clients) exit(0);
    if (poll(fdv,fdvn,INFTIM) < 0)
     { if (errno == EINTR) continue;
       fprintf(stderr,"%s; poll: %s\n",__progname,strerror(errno));
       exit(1);
     }
    if (fdv[0].revents & (POLLIN|POLLRDNORM|POLLRDBAND|POLLPRI))
     { ctl_read();
       continue;
     }
    if ((death >= 0) && (fdv[1].revents & (POLLIN|POLLRDNORM|POLLRDBAND|POLLPRI)))
     { close(death);
       death = -1;
       continue;
     }
    for (c=clients;c;c=c->link)
     { if (fdv[c->fdx].revents & (POLLIN|POLLRDNORM|POLLRDBAND|POLLPRI))
	{ client_read(c);
	}
     }
  }
}

static int devnull(void)
{
 static int fd = -1;

 if (fd < 0) fd = open("/dev/null",O_RDWR,0);
 return(fd);
}

static int mypipe(int *x)
{
 char buf[32];
 int rv;

 rv = pipe(x);
 if (rv < 0) return(-1);
 sprintf(&buf[0],"%d %d",x[0],x[1]);
 write(devnull(),&buf[0],strlen(&buf[0]));
 return(rv);
}
#define pipe(x) mypipe(x)

static int mysocketpair(int f, int d, int p, int *x)
{
 char buf[32];
 int rv;

 rv = socketpair(f,d,p,x);
 if (rv < 0) return(-1);
 sprintf(&buf[0],"%d %d",x[0],x[1]);
 write(devnull(),&buf[0],strlen(&buf[0]));
 return(rv);
}
#define socketpair(f,d,p,x) mysocketpair(f,d,p,x)

static void fork_tracer(void)
{
 int s[2];
 int p[2];
 pid_t kid;

 if (socketpair(AF_LOCAL,SOCK_DGRAM,0,&s[0]) < 0)
  { fprintf(stderr,"%s: socketpair AF_LOCAL SOCK_DGRAM: %s\n",__progname,strerror(errno));
    exit(1);
  }
 if (pipe(&p[0]) < 0)
  { fprintf(stderr,"%s: pipe: %s\n",__progname,strerror(errno));
    exit(1);
  }
 kid = fork();
 if (kid < 0)
  { fprintf(stderr,"%s: fork: %s\n",__progname,strerror(errno));
    exit(1);
  }
 if (kid == 0)
  { close(p[1]);
    /* don't close s[0]; let trace filter merger process hold it open
       to prevent ECONNRESET recv errors (we'd rather get death notification
       via the death pipe, here called p). */
    run_tracer(s[1],p[0]);
    exit(0);
  }
 close(p[0]);
 /* drop p[1], let it be closed when we exit - but make sure any future
    children forked to run other programs don't inherit it! */
 fcntl(p[1],F_SETFD,1);
 close(s[1]);
 trace_ctl = s[0];
}

static void send_tracer(int fd)
{
 struct msghdr mh;
 struct iovec iov;
#ifdef SCM_RIGHTS
 struct cmsghdr cmh;
 char cbuf[sizeof(struct cmsghdr)+sizeof(int)];
#endif

 mh.msg_name = 0;
 mh.msg_namelen = 0;
 iov.iov_base = (void *)&iov; /* XXX, cast is for SunOS */
 iov.iov_len = 1;
 mh.msg_iov = &iov;
 mh.msg_iovlen = 1;
#ifdef SCM_RIGHTS
 cmh.cmsg_len = sizeof(cbuf);
 cmh.cmsg_level = SOL_SOCKET;
 cmh.cmsg_type = SCM_RIGHTS;
 bcopy(&cmh,&cbuf[0],sizeof(cmh));
 bcopy(&fd,&cbuf[sizeof(cmh)],sizeof(fd));
 mh.msg_control = &cbuf[0];
 mh.msg_controllen = sizeof(cbuf);
 mh.msg_flags = 0;
#else
 mh.msg_accrights = (void *)&fd;
 mh.msg_accrightslen = sizeof(int);
#endif
 if (sendmsg(trace_ctl,&mh,0) < 0)
  { fprintf(stderr,"%s: can't send trace fd to tracer: %s\n",__progname,strerror(errno));
    exit(1);
  }
 close(fd);
}

static void tracefilter(int rfd, int wfd, const char *tag)
{
 int i;
 int p1[2];
 int p2[2];
 int p3[2];
 unsigned char buf[WORKSIZE];
 int flags;
#define LIVE_R  1
#define LIVE_W  2

 if (trace_ctl < 0) fork_tracer();
 if ((pipe(&p1[0]) < 0) || (pipe(&p2[0]) < 0) || (pipe(&p3[0]) < 0))
  { fprintf(stderr,"%s: pipe: %s\n",__progname,strerror(errno));
    exit(1);
  }
 switch (fork())
  { case -1:
       fprintf(stderr,"%s: can't fork: %s\n",__progname,strerror(errno));
       exit(1);
       break;
    case 0:
       break;
    default:
       dup2(p1[0],rfd);
       dup2(p2[1],wfd);
       close(p1[0]);
       close(p1[1]);
       close(p2[0]);
       close(p2[1]);
       send_tracer(p3[0]);
       close(p3[1]);
       return;
       break;
  }
 if (p3[1] != 2)
  { dup2(p3[1],2);
    close(p3[1]);
  }
 for (i=getdtablesize()-1;i>=0;i--)
  { if ((i != 2) && (i != p1[1]) && (i != p2[0]) && (i != rfd) && (i != wfd))
     { close(i);
     }
  }
 flags = LIVE_R | LIVE_W;
 while (flags & (LIVE_R|LIVE_W))
  { fd_set fds;
    int n;
    FD_ZERO(&fds);
    if (flags & LIVE_W) FD_SET(p2[0],&fds);
    if (flags & LIVE_R) FD_SET(rfd,&fds);
    n = select(FD_SETSIZE,&fds,(fd_set *)0,(fd_set *)0,(struct timeval *)0);
    if (n < 0)
     { if (errno != EINTR)
	{ fprintf(stderr,"%s: select error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       continue;
     }
    if ((flags & LIVE_W) && FD_ISSET(p2[0],&fds))
     { n = read(p2[0],&buf[0],sizeof(buf));
       if (n < 0)
	{ fprintf(stderr,"%s: read error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       if (n == 0)
	{ fprintf(stderr,"%s [closed >>]\n",tag);
	  close(p2[0]);
	  close(wfd);
	  flags &= ~LIVE_W;
	}
       else
	{ trace_print(tag,">>",&buf[0],n);
	  write(wfd,&buf[0],n);
	}
     }
    if ((flags & LIVE_R) && FD_ISSET(rfd,&fds))
     { n = read(rfd,&buf[0],sizeof(buf));
       if (n < 0)
	{ fprintf(stderr,"%s: read error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       if (n == 0)
	{ fprintf(stderr,"%s [closed <<]\n",tag);
	  close(rfd);
	  close(p1[1]);
	  flags &= ~LIVE_R;
	}
       else
	{ trace_print(tag,"<<",&buf[0],n);
	  write(p1[1],&buf[0],n);
	}
     }
  }
 exit(0);
#undef LIVE_R
#undef LIVE_W
}

static void xfilter(int rfd, int wfd, const char *tag)
{
 int i;
 int p1[2];
 int p2[2];
 unsigned char binbuf[256];
 unsigned char hexbuf[512];
 unsigned char oddsave;
 int flags;
#define SHUT_R  1
#define SHUT_W  2
#define ODDBYTE 4

 if ((pipe(&p1[0]) < 0) || (pipe(&p2[0]) < 0))
  { fprintf(stderr,"%s: can't create pipe: %s\n",__progname,strerror(errno));
    exit(1);
  }
 switch (fork())
  { case -1:
       fprintf(stderr,"%s: can't fork: %s\n",__progname,strerror(errno));
       exit(1);
       break;
    case 0:
       break;
    default:
       dup2(p1[0],rfd);
       close(p1[1]);
       close(p2[0]);
       dup2(p2[1],wfd);
       return;
       break;
  }
 for (i=getdtablesize()-1;i>=0;i--)
  { if ((i != 2) && (i != p1[1]) && (i != p2[0]) && (i != rfd) && (i != wfd))
     { close(i);
     }
  }
 flags = 0;
 while (1)
  { fd_set fds;
    int n;
    if ((flags & (SHUT_R|SHUT_W)) == (SHUT_R|SHUT_W)) exit(0);
    FD_ZERO(&fds);
    if (! (flags & SHUT_W)) FD_SET(p2[0],&fds);
    if (! (flags & SHUT_R)) FD_SET(rfd,&fds);
    n = select(FD_SETSIZE,&fds,(fd_set *)0,(fd_set *)0,(struct timeval *)0);
    if (n < 0)
     { if (errno != EINTR)
	{ fprintf(stderr,"%s: select error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       continue;
     }
    if (FD_ISSET(p2[0],&fds))
     { n = read(p2[0],(char *)&binbuf[0],sizeof(binbuf));
       if (n < 0)
	{ fprintf(stderr,"%s: bin read error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       if (n == 0)
	{ close(wfd);
	  flags |= SHUT_W;
	}
       else
	{ bin2hex(&binbuf[0],&hexbuf[0],n);
	  if (flag_trace) trace_print(tag,"(enc)>>",&hexbuf[0],n*2);
	  write(wfd,(char *)&hexbuf[0],n*2);
	}
     }
    if (FD_ISSET(rfd,&fds))
     { if (flags & ODDBYTE)
	{ n = read(rfd,(char *)&hexbuf[1],sizeof(hexbuf)-1);
	  if (n < 0)
	   { fprintf(stderr,"%s: hex read error: %s\n",__progname,strerror(errno));
	     exit(1);
	   }
	  if (n == 0)
	   { close(p1[1]);
	     flags |= SHUT_R;
	   }
	  else
	   { n ++;
	     hexbuf[0] = oddsave;
	     flags &= ~ODDBYTE;
	   }
	}
       else
	{ n = read(rfd,(char *)&hexbuf[0],sizeof(hexbuf));
	  if (n < 0)
	   { fprintf(stderr,"%s: hex read error: %s\n",__progname,strerror(errno));
	     exit(1);
	   }
	  if (n == 0)
	   { close(p1[1]);
	     flags |= SHUT_R;
	   }
	}
       if (n)
	{ if (n & 1)
	   { oddsave = hexbuf[--n];
	     flags |= ODDBYTE;
	   }
	  n /= 2;
	  if (flag_trace) trace_print(tag,"(enc)<<",&hexbuf[0],n*2);
	  hex2bin(&hexbuf[0],&binbuf[0],n);
	  write(p1[1],(char *)&binbuf[0],n);
	}
     }
  }
#undef SHUT_R
#undef SHUT_W
#undef ODDBYTE
}

static void newremote(char *arg)
{
 REMOTE *r;
 char *slash;
 char *at;
 char *colon;
 char *bang;

 r = NEW(REMOTE);
 r->rfd = -1;
 r->wfd = -1;
 r->link = remotes;
 remotes = r;
 r->fullarg = copyofstr(arg);
 slash = index(arg,'/');
 at = index(arg,'@');
 colon = index(arg,':');
 bang = index(arg,'!');
 /*
	can have  user   machine   program   directory
	    @       no     yes       yes     yes
	    !      yes      no       yes     yes
	    :       no      no        no     only after a /
	    /       no      no       yes     yes

	user @ machine ! program : directory	exist(@!:) and (@ < ! < :) and ((no /) or (! < /))
	       machine ! program : directory	exist(!:) and (! < :) and ((no /) or (! < /))
	user @ machine           : directory	exist(@:) and (@ < :) and ((no /) or (: < /))
	       machine           : directory	exist(:) and ((no /) or (: < /))
				   directory	none of the above
 */
 if (at && bang && colon && (at < bang) && (bang < colon) && (!slash || (bang < slash)))
  { *at = '\0';
    *bang = 0;
    *colon = 0;
    r->user = arg;
    r->machine = at + 1;
    r->program = bang + 1;
    r->directory = colon + 1;
  }
 else if (bang && colon && (bang < colon) && (!slash || (bang < slash)))
  { *bang = 0;
    *colon = 0;
    r->user = 0;
    r->machine = arg;
    r->program = bang + 1;
    r->directory = colon + 1;
  }
 else if (at && colon && (at < colon) && (!slash || (colon < slash)))
  { *at = 0;
    *colon = 0;
    r->user = arg;
    r->machine = at + 1;
    r->program = 0;
    r->directory = colon + 1;
  }
 else if (colon && (!slash || (colon < slash)))
  { *colon = 0;
    r->user = 0;
    r->machine = arg;
    r->program = 0;
    r->directory = colon + 1;
  }
 else
  { r->user = 0;
    r->machine = 0;
    r->program = 0;
    r->directory = arg;
  }
 r->rsh = currsh;
 r->ignnx = flag_nextinx;
 flag_nextinx = 0;
}

static void reaper(int sig __attribute__((__unused__)))
{
 while (wait3(0,WNOHANG,0) > 0) ;
}

static void sgetnts(void)
{
 int x;
 int c;

 x = 0;
 while (1)
  { c = getchar();
    if (c == EOF)
     { fprintf(stderr,"%s: unexpected EOF (in sgetnts)\n",__progname);
       exit(1);
     }
    slavework[x] = c;
    if (c == 0) return;
    if (x < WORKSIZE-1) x ++;
  }
}

static void etc_init(ENTRY_TCONC *etc)
{
 etc->tail = &etc->list;
}

static void etc_add(ENTRY_TCONC *etc, ENTRY *e)
{
 *etc->tail = e;
 etc->tail = &e->link;
}

static void etc_end(ENTRY_TCONC *etc)
{
 *etc->tail = 0;
}

static void etc_append(ENTRY_TCONC *etc, ENTRY *e)
{
 if (! e) return;
 *etc->tail = e;
 while (e->link) e = e->link;
 etc->tail = &e->link;
}

static void fork_filter(int ifd, int ofd, const char *argv0, ...)
{
 va_list ap;
 int p[2];
 int na;
 const char **av;
 void (*fn)(void);
 int kid;
 int i;

 if ((ifd < 0) && (ofd < 0)) panic("fork_filter: both fds negative");
 if ((ifd >= 0) && (ofd >= 0)) panic("fork_filter: neither fd negative");
 if (pipe(p) < 0) panic("fork_filter: can't make pipe: %s",strerror(errno));
 if (argv0)
  { va_start(ap,argv0);
    for (na=1;va_arg(ap,const char *);na++) ;
    va_end(ap);
    av = malloc((na+1)*sizeof(const char *));
    av[0] = argv0;
    va_start(ap,argv0);
    for (na=1;(av[na]=va_arg(ap,const char *));na++) ;
    va_end(ap);
  }
 else
  { typedef void (*foo)(void); /* stupid varargs misfeature */
    va_start(ap,argv0);
    fn = va_arg(ap,foo);
    va_end(ap);
  }
 while (wait3(0,WNOHANG,0) > 0) ;
 kid = fork();
 if (kid)
  { if (ifd < 0)
     { close(ofd);
       close(p[0]);
       dup2(p[1],ofd);
       close(p[1]);
     }
    else
     { close(ifd);
       close(p[1]);
       dup2(p[0],ifd);
       close(p[0]);
     }
    return;
  }
 for (i=getdtablesize()-1;i>2;i--)
  { if ((i != ifd) && (i != ofd) && (i != p[0]) && (i != p[1])) close(i);
  }
 if (ifd < 0)
  { if (ofd != 1)
     { dup2(ofd,1);
       close(ofd);
     }
    if (p[0] != 0)
     { dup2(p[0],0);
       close(p[0]);
     }
    close(p[1]);
  }
 else
  { if (ifd != 0)
     { dup2(ifd,0);
       close(ifd);
     }
    if (p[1] != 1)
     { dup2(p[1],1);
       close(p[1]);
     }
    close(p[0]);
  }
 if (argv0)
  { execvp(argv0,(const void *)av);
    perror(argv0);
  }
 else
  { (*fn)();
  }
 exit(1);
}

static void slave_rm(void); /* forward */

/* always sets errno */
static void slave_rm_r(void)
{
 ENTRY_TCONC etc;
 ENTRY *e;
 DIR *dir;
 struct direct *d;
 int err;
 int len;

 err = 0;
 etc_init(&etc);
 dir = opendir(&slavetemp[0]);
 while ((d=readdir(dir)))
  { if ( (d->d_name[0] == '.') &&
	 ( (d->d_namlen == 1) ||
	   ( (d->d_name[1] == '.') &&
	     (d->d_namlen == 2) ) ) ) continue;
    e = NEW(ENTRY);
    e->name = copyofstr(&d->d_name[0]);
    etc_add(&etc,e);
  }
 closedir(dir);
 etc_end(&etc);
 len = strlen(&slavetemp[0]);
 slavetemp[len++] = '/';
 while (etc.list)
  { e = etc.list;
    etc.list = e->link;
    strcpy(&slavetemp[len],e->name);
    slave_rm();
    if (err == 0) err = errno;
    free(e->name);
    OLD(e);
  }
 slavetemp[len-1] = '\0';
 errno = err;
}

/* unlink(2) on a directory either produces EPERM or orphans the stuff in
   the directory, depending on system; it doesn't necessarily do anything
   sensible and distinctive, like EISDIR, so we have to waste a syscall.... */

/* always sets errno */
static void slave_rm(void)
{
 errno = 0;
 if (rmdir(&slavetemp[0]) == 0) return;
 switch (errno)
  { case ENOTEMPTY:
       slave_rm_r();
       if ((errno == 0) && (rmdir(&slavetemp[0]) == 0)) errno = 0;
       break;
    case ENOTDIR:
       if (unlink(&slavetemp[0]) == 0) errno = 0;
       break;
    default:
       break;
  }
}

static int makenulls(int fd, unsigned long int bytes)
{
 int n;
 int r;
 int w;
 char rbuf[512];

 while (bytes > 0)
  { n = (bytes < 512) ? bytes : 512;
    r = read(fd,&rbuf[0],n);
    if (r < 0) return(-1);
    if (r == 0) return(lseek(fd,bytes,L_INCR));
    if (bcmp(&rbuf[0],&zeroblk[0],r))
     { if (lseek(fd,-r,L_INCR) < 0) return(-1);
       w = write(fd,&zeroblk[0],r);
       if (w < 0) return(-1);
       if (w < r) return(EIO);
     }
    bytes -= r;
  }
 return(0);
}

static void swrite(int skip, const void *buf, int nb)
{
 int w;

 if ((skip > 0) && (makenulls(1,skip*512) < 0))
  { fprintf(stderr,"%s: sparsewrite seek error: %s\n",__progname,strerror(errno));
    exit(1);
  }
 if (nb > 0)
  { w = write(1,buf,nb);
    if (w != nb)
     { if (w < 0)
	{ fprintf(stderr,"%s: sparsewrite write error: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       fprintf(stderr,"%s: sparsewrite write: tried %d, got %d\n",__progname,nb,w);
       exit(1);
     }
  }
}

/* not all ftruncate()s will grow short files, grr! */
static void setsize(int fd, unsigned long int size)
{
 struct stat stb;

 ftruncate(fd,size);
 fstat(fd,&stb);
 if (stb.st_size < size)
  { lseek(fd,stb.st_size+(1<<20),L_SET);
    write(fd,"",1);
    ftruncate(fd,size);
  }
}

static void sparsewrite(void)
{
 int inbuf;
 int base;
 int n;
 int skip;
 int w0;
 unsigned long int size;

 inbuf = 0;
 skip = 0;
 size = 0;
 while (1)
  { if (inbuf == 0) base = 0;
    n = read(0,&slavework[base],WORKSIZE-base);
    if (n < 0)
     { fprintf(stderr,"%s: sparsewrite read error: %s\n",__progname,strerror(errno));
       exit(1);
     }
    if (n == 0)
     { swrite(skip,&slavework[base],inbuf);
       setsize(1,size);
       exit(0);
     }
    size += n;
    inbuf += n;
    w0 = -1;
    while (inbuf >= 512)
     { if (bcmp(&slavework[base],&zeroblk[0],512))
	{ if (w0 < 0) w0 = base;
	}
       else
	{ if (w0 >= 0)
	   { swrite(skip,&slavework[w0],base-w0);
	     skip = 0;
	     w0 = -1;
	   }
	  skip ++;
	}
       inbuf -= 512;
       base += 512;
     }
    if (w0 >= 0)
     { swrite(skip,&slavework[w0],base-w0);
       skip = 0;
     }
  }
}

static void slavepathcheck(void)
{
 char *dp;
 char *wp;

 wp = &slavework[0];
 if ((wp[0] == '.') && (wp[1] == '.') && ((wp[2] == '/') || (wp[2] == '\0')))
  {
badpath:;
    fprintf(stderr,"%s: bad path %s from master\n",__progname,&slavework[0]);
    exit(1);
  }
 while (1)
  { dp = index(wp,'.');
    if (! dp) break;
    if ( (dp[-1] == '/') &&
	 (dp[1] == '.') &&
	 ((dp[2] == '\0') || (dp[2] == '/')) ) goto badpath;
    wp = dp + 1;
  }
}

static void slave_D(void)
{
 DIR *dir;
 struct direct *d;
 int l;
 int ll;
 struct stat stb;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 dir = opendir(&slavetemp[0]);
 if (dir == 0)
  { putchar('\0');
    return;
  }
 l = strlen(slavetemp);
 slavetemp[l++] = '/';
 while ((d=readdir(dir)))
  { if ( (d->d_ino == 0) ||
	 ( (d->d_name[0] == '.') &&
	   ( (d->d_name[1] == '\0') ||
	     ( (d->d_name[1] == '.') &&
	       (d->d_name[2] == '\0') ) ) ) ) continue;
    strcpy(&slavetemp[l],d->d_name);
    if ((flag_follow?stat(&slavetemp[0],&stb):lstat(&slavetemp[0],&stb)) < 0) continue;
    fputs(d->d_name,stdout);
    putchar(0);
    printf("%d %d %d",(int)stb.st_mode,(int)stb.st_uid,(int)stb.st_gid);
    if (flag_mtimes) printf(" %ld",(long int)stb.st_mtime);
    if (flag_links) printf(" %d",(int)stb.st_nlink);
    switch (stb.st_mode & S_IFMT)
     { case S_IFDIR:
	  break;
       case S_IFCHR:
       case S_IFBLK:
	  printf(" %d %d",major(stb.st_rdev),minor(stb.st_rdev));
	  break;
       case S_IFREG:
	  printf(" %lu",(unsigned long int)stb.st_size);
	  break;
       case S_IFLNK:
	  putchar(0);
	  ll = readlink(&slavetemp[0],&slavework[0],sizeof(slavework));
	  if (ll < 0) ll = 0;
	  fwrite(&slavework[0],1,ll,stdout);
	  break;
       case S_IFSOCK:
	  break;
#ifdef S_IFIFO
       case S_IFIFO:
	  break;
#endif
     }
    putchar(0);
  }
 closedir(dir);
 putchar(0);
}

static void slave_I(void)
{
 int fd;
 int cmd;
 int nb;
 int nr;
 long int off;
 long int curoff;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 fd = open(&slavetemp[0],O_RDONLY,0);
 if (flag_debug) fprintf(stderr,"slave_I entering command loop\n");
 curoff = 0;
 while (1)
  { fflush(stdout);
    cmd = getchar();
    switch (cmd)
     { default:
	  fprintf(stderr,"%s: protocol error, unknown slave I subcommand %d (%c)\n",__progname,cmd,cmd);
	  exit(1);
	  break;
       case EOF:
	  fprintf(stderr,"%s: protocol error, unexpected EOF (slave_I)\n",__progname);
	  exit(1);
	  break;
       case 'c':
	  goto out; /* aka break 2 */
	  break;
       case 'r':
	  sgetnts();
	  sscanf(&slavework[0],"%d%ld",&nb,&off);
	  if (nb > WORKSIZE)
	   { fprintf(stderr,"%s: protocol error, nb too big (%d > %d) in slave R r\n",__progname,nb,WORKSIZE);
	     exit(1);
	   }
	  if (off != curoff) lseek(fd,off,L_SET);
	  curoff = off;
	  nr = read(fd,&slavetemp[0],nb);
	  if (nr <= 0)
	   { printf("%d",COPYBLK_HOLE);
	     putchar('\0');
	   }
	  else
	   { if (nr < nb) bzero(&slavetemp[nr],nb-nr);
	     if (bcmp(&slavetemp[0],&zeroblk[0],nb))
	      { printf("%d",COPYBLK_DATA);
		putchar('\0');
		fwrite(&slavetemp[0],1,nb,stdout);
	      }
	     else
	      { printf("%d",COPYBLK_HOLE);
		putchar('\0');
	      }
	   }
	  curoff += (nr >= 0) ? nr : 0;
	  break;
     }
  }
out:;
 if (fd >= 0) close(fd);
}

static void slave_K(void)
{
 int fd;
 void *md5;
 int nb;
 unsigned long int from;
 unsigned long int to;
 int n;
 char cksum[16];

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%lu %lu",&from,&to);
 if (flag_debug) fprintf(stderr,"slave_K %s %lu %lu\n",&slavetemp[0],from,to);
 fd = open(&slavetemp[0],O_RDONLY,0);
 if (fd < 0)
  { printf("O%d",errno);
    putchar('\0');
  }
 else
  { md5 = md5_init();
    lseek(fd,from,SEEK_SET);
    to -= from;
    while (1)
     { n = ulmin(to,WORKSIZE);
       if (n < 1)
	{ md5_result(md5,&cksum[0]);
	  printf("K");
	  fwrite(&cksum[0],1,16,stdout);
	  break;
	}
       nb = n ? read(fd,&slavetemp[0],n) : 0;
       if (nb < 0)
	{ printf("R%d",errno);
	  putchar('\0');
	  break;
	}
       if (nb == 0)
	{ printf("R0");
	  putchar('\0');
	  break;
	}
       md5_process_bytes(md5,&slavetemp[0],nb);
       to -= nb;
     }
    close(fd);
  }
}

static void slave_R(void)
{
 int fd;
 int cmd;
 int nb;
 int nr;
 long int off;
 long int curoff;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 fd = open(&slavetemp[0],O_RDONLY,0);
 if (flag_debug) fprintf(stderr,"slave_R entering command loop\n");
 curoff = 0;
 while (1)
  { fflush(stdout);
    cmd = getchar();
    switch (cmd)
     { default:
	  fprintf(stderr,"%s: protocol error, unknown slave R subcommand %d (%c)\n",__progname,cmd,cmd);
	  exit(1);
	  break;
       case EOF:
	  fprintf(stderr,"%s: protocol error, unexpected EOF (slave_R)\n",__progname);
	  exit(1);
	  break;
       case 'c':
	  goto out; /* aka break 2 */
	  break;
       case 'r':
	  sgetnts();
	  sscanf(&slavework[0],"%d%ld",&nb,&off);
	  if (nb > WORKSIZE)
	   { fprintf(stderr,"%s: protocol error, nb too big (%d > %d) in slave R r\n",__progname,nb,WORKSIZE);
	     exit(1);
	   }
	  if (off != curoff) lseek(fd,curoff,L_SET);
	  curoff = off;
	  nr = read(fd,&slavetemp[0],nb);
	  if (nr <= 0)
	   { curoff = -1;
	     fwrite(&zeroblk[0],1,nb,stdout);
	   }
	  else
	   { curoff += nr;
	     if (nr < nb)
	      { fwrite(&slavetemp[0],1,nr,stdout);
		fwrite(&zeroblk[0],1,nb-nr,stdout);
	      }
	     else
	      { fwrite(&slavetemp[0],1,nb,stdout);
	      }
	   }
	  break;
     }
  }
out:;
 if (fd >= 0) close(fd);
}

static void slave_d(void)
{
 int mode;
 int uid;
 int gid;
 struct stat stb;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d%d%d",&mode,&uid,&gid);
 if (flag_debug) fprintf(stderr,"slave_d %s %o %d %d\n",&slavetemp[0],mode,uid,gid);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if ( (mkdir(&slavetemp[0],mode) == 0) &&
      (stat(&slavetemp[0],&stb) == 0) &&
      ( (((stb.st_mode^mode) & (flag_sgiddir?05777:07777)) == 0) ||
	(chmod(&slavetemp[0],flag_sgiddir?((stb.st_mode&02000)|(mode&05777)):mode) == 0) ) &&
      ( flag_noown ||
	( (chown(&slavetemp[0],-1,gid) == 0) &&
	  (chown(&slavetemp[0],uid,-1) == 0) ) ) ) errno = 0;
 printf("%d",errno);
 putchar('\0');
}

static void slave_f(void)
{
 int mode;
 int uid;
 int gid;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d%d%d",&mode,&uid,&gid);
 if (flag_debug) fprintf(stderr,"slave_f %s %o %d %d\n",&slavetemp[0],mode,uid,gid);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
#ifdef S_IFIFO
 if ( (mkfifo(&slavetemp[0],mode) == 0) &&
      ( flag_noown || 
	( (chown(&slavetemp[0],-1,gid) == 0) &&
	  (chown(&slavetemp[0],uid,-1) == 0) ) ) ) errno = 0;
#else
 errno = ENODEV;
#endif
 printf("%d",errno);
 putchar('\0');
}

static int open_to_update(const char *path, int mode, int *remodp)
{
 int rechmod;
 int fd;

 rechmod = -1;
 mode &= 07777;
 fd = open(path,O_RDWR|O_CREAT,mode);
 if (fd < 0)
  { fd = open(path,O_WRONLY|O_CREAT,mode);
    if (fd < 0)
     { struct stat stb;
       if (errno != EACCES) return(-1);
       if ( (stat(path,&stb) < 0) ||
	    (chmod(path,0600) < 0) )
	{ errno = EACCES;
	  return(-1);
	}
       rechmod = stb.st_mode & 07777;
       fd = open(path,O_RDWR,0);
       if (fd < 0)
	{ int e;
	  e = errno;
	  chmod(path,rechmod);
	  errno = e;
	  return(-1);
	}
     }
  }
 *remodp = rechmod;
 return(fd);
}

static void slave_i(void)
{
 int mode;
 int uid;
 int gid;
 unsigned long int size;
 unsigned long int from;
 unsigned long int to;
 int fd;
 unsigned long int left;
 int blkleft;
 int blkoff;
 int flg;
 int n;
 int r;
 int tempfill;
 int skip;
 int w0;
 int writearg;
 int writerv;
 int finalerr;
 int rechmod;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d%d%d%lu%lu%lu",&mode,&uid,&gid,&size,&from,&to);
 if (flag_debug) fprintf(stderr,"slave_i %s %o %d %d %lu %lu %lu\n",&slavetemp[0],mode&07777,uid,gid,size,from,to);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 fd = open_to_update(&slavetemp[0],mode&07777,&rechmod);
 if (fd < 0)
  { printf("%d",errno);
    putchar('\0');
    return;
  }
 if ( ( !flag_noown &&
	( (fchown(fd,-1,gid) < 0) ||
	  (fchown(fd,uid,-1) < 0) ) ) ||
      (lseek(fd,from,SEEK_SET) < 0) )
  { printf("%d",errno);
    putchar('\0');
    if (rechmod >= 0) fchmod(fd,rechmod);
    close(fd);
    return;
  }
 putchar('0');
 putchar('\0');
 fflush(stdout);
 blkleft = 0;
 blkoff = 0;
 tempfill = 0;
 left = to - from;
 skip = 0;
 finalerr = 0;
#define WRITE(buf,nb) do { if ((writearg=(nb)) && (writerv=write(fd,(buf),writearg)) != writearg) goto writeerr; } while (0)
 while ((left > 0) || (blkleft > 0))
  { if (0)
     {
writeerr:;
       finalerr = (writerv < 0) ? errno : EIO;
       if (rechmod >= 0) fchmod(fd,rechmod);
       close(fd);
       fd = -1;
     }
    if (0)
     {
fderr:;
       finalerr = errno;
       if (rechmod >= 0) fchmod(fd,rechmod);
       close(fd);
       fd = -1;
     }
    if (blkleft < 1)
     { sgetnts();
       if (sscanf(&slavework[0],"%d%d",&n,&flg) != 2)
	{ fprintf(stderr,"%s: protocol error (can't read block size and type)\n",__progname);
	  exit(1);
	}
       blkleft += n;
       blkoff = 0;
       left -= n;
     }
    switch (flg)
     { case COPYBLK_DATA:
	  n = WORKSIZE - tempfill;
	  if (blkleft < n) n = blkleft;
	  r = fread(&slavework[tempfill],1,n,stdin);
	  if (r != n)
	   { fprintf(stderr,"%s: protocol error (can't read block data)\n",__progname);
	     exit(1);
	   }
	  blkleft -= n;
	  blkoff += n;
	  if (fd < 0) continue;
	  n += tempfill;
	  r = 0;
	  w0 = -1;
	  while (n-r >= 512)
	   { if (flag_nosparse || bcmp(&slavework[r],&zeroblk[0],512))
	      { if (w0 < 0) w0 = r;
		if (skip > 0)
		 { if (makenulls(fd,skip) < 0) goto fderr;
		   skip = 0;
		 }
	      }
	     else
	      { if (w0 >= 0)
		 { WRITE(&slavework[w0],r-w0);
		   w0 = -1;
		 }
		skip += 512;
	      }
	     r += 512;
	   }
	  if (w0 >= 0) WRITE(&slavework[w0],r-w0);
	  tempfill = n - r;
	  if (tempfill && (r > 0)) bcopy(&slavework[r],&slavework[0],tempfill);
	  break;
       case COPYBLK_HOLE:
	  n = blkleft;
	  blkleft = 0;
	  blkoff += n;
	  if (fd < 0) continue;
	  if (tempfill+n < 512)
	   { bzero(&slavework[tempfill],n);
	     tempfill += n;
	   }
	  else
	   { if (flag_nosparse)
	      { if (tempfill > 0)
		 { bzero(&slavework[tempfill],512-tempfill);
		   WRITE(&slavework[0],512);
		   n -= 512 - tempfill;
		 }
		while (n >= WORKSIZE)
		 { WRITE(&zeroblk[0],WORKSIZE);
		   n -= WORKSIZE;
		 }
		if (n >= 512)
		 { WRITE(&zeroblk[0],n&~511);
		   n &= 511;
		 }
		if (n > 0) bzero(&slavework[0],n);
		tempfill = n;
	      }
	     else
	      { if (tempfill)
		 { if (bcmp(&slavework[0],&zeroblk[0],tempfill))
		    { bzero(&slavework[tempfill],512-tempfill);
		      if (skip > 0)
		       { if (makenulls(fd,skip) < 0) goto fderr;
			 skip = 0;
		       }
		      WRITE(&slavework[0],512);
		      n -= 512 - tempfill;
		    }
		   else
		    { n += tempfill;
		    }
		 }
		skip += n;
		tempfill = skip % 512;
		if (tempfill)
		 { skip -= tempfill;
		   bzero(&slavework[0],tempfill);
		 }
	      }
	   }
	  break;
     }
  }
 if (tempfill > 0)
  { if (skip > 0)
     { if (makenulls(fd,skip) < 0)
	{ finalerr = errno;
	  if (rechmod >= 0) fchmod(fd,rechmod);
	  close(fd);
	  fd = -1;
	}
       skip = 0;
     }
    if (fd >= 0) WRITE(&slavework[0],tempfill);
  }
 if (fd >= 0)
  { setsize(fd,size);
    if (rechmod >= 0) fchmod(fd,rechmod);
    close(fd);
    putchar('0');
  }
 else
  { printf("%d",finalerr);
  }
#undef WRITE
 putchar('\0');
}

static void slave_l(void)
{
 int mode;
 int uid;
 int gid;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d%d%d",&mode,&uid,&gid);
 sgetnts();
 if (flag_debug) fprintf(stderr,"slave_l %s -> %s\n",&slavetemp[0],&slavework[0]);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if ( (symlink(&slavework[0],&slavetemp[0]) == 0) &&
      (lchmod(&slavetemp[0],mode) == 0) &&
      ( flag_noown ||
	( (lchown(&slavetemp[0],-1,gid) == 0) &&
	  (lchown(&slavetemp[0],uid,-1) == 0) ) ) ) errno = 0;
 printf("%d",errno);
 putchar('\0');
}

static void slave_m(void)
{
 int mode;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d",&mode);
 if (flag_debug) fprintf(stderr,"slave_m %s %o\n",&slavetemp[0],mode&07777);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if (flag_sgiddir)
  { struct stat stb;
    if ((stat(&slavetemp[0],&stb) == 0) && ((stb.st_mode & S_IFMT) == S_IFDIR))
     { mode = (mode & 05777) | (stb.st_mode & 02000);
     }
    if (lchmod(&slavetemp[0],mode) == 0) errno = 0;
  }
 else
  { if (lchmod(&slavetemp[0],mode) == 0) errno = 0;
  }
 printf("%d",errno);
 putchar('\0');
}

static void slave_n(void)
{
 int mode;
 int uid;
 int gid;
 int rdev;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d %d %d %d",&mode,&uid,&gid,&rdev);
 if (flag_debug) fprintf(stderr,"slave_n %s %o %d %d (%d,%d)\n",&slavetemp[0],mode,uid,gid,major(rdev),minor(rdev));
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if ( (mknod(&slavetemp[0],mode,rdev) == 0) &&
      ( flag_noown ||
	( (chown(&slavetemp[0],-1,gid) == 0) &&
	  (chown(&slavetemp[0],uid,-1) == 0) ) ) ) errno = 0;
 printf("%d",errno);
 putchar('\0');
}

static void slave_o(void)
{
 int uid;
 int gid;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d %d",&uid,&gid);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if (flag_debug) fprintf(stderr,"slave_o %s %d %d\n",&slavetemp[0],uid,gid);
 if ( flag_noown ||
      ( (lchown(&slavetemp[0],-1,gid) == 0) &&
	(lchown(&slavetemp[0],uid,-1) == 0) ) ) errno = 0;
 printf("%d",errno);
 putchar('\0');
}

static void slave_r(void)
{
 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 if (flag_debug) fprintf(stderr,"slave_r %s\n",&slavetemp[0]);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 slave_rm();
 printf("%d",errno);
 putchar('\0');
}

static void slave_t(void)
{
 unsigned long int sz;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%lu",&sz);
 if (flag_debug) fprintf(stderr,"slave_t %s %lu\n",&slavetemp[0],sz);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 /* not all truncate()s will grow short files, grr! */
 if (truncate(&slavetemp[0],sz) >= 0)
  { struct stat stb;
    if (stat(&slavetemp[0],&stb) >= 0)
     { if (stb.st_size < sz)
	{ int fd;
	  fd = open(&slavetemp[0],O_RDWR,0);
	  if (fd >= 0)
	   { lseek(fd,sz+(1UL<<20),L_SET);
	     write(fd,"",1);
	     ftruncate(fd,sz);
	     close(fd);
	     errno = 0;
	   }
	}
       else if (stb.st_size > sz)
	{ errno = ERANGE;
	}
       else
	{ errno = 0;
	}
     }
  }
 printf("%d",errno);
 putchar('\0');
}

static void slave_u(void)
{
 struct timeval times[2];
 struct stat stb;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%ld %ld",&times[1].tv_sec,&times[1].tv_usec);
 stat(&slavetemp[0],&stb);
 times[0].tv_sec = stb.st_atime;
 times[0].tv_usec = 0;
 if (flag_debug)
  { time_t tmp;
    tmp = stb.st_atime;
    fprintf(stderr,"slave_u %s %.24s\n",&slavetemp[0],ctime(&tmp));
  }
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 if (utimes(&slavetemp[0],&times[0]) == 0) errno = 0;
 printf("%d",errno);
 putchar('\0');
}

static void slave_z(void)
{
 if (! flag_gzip)
  { fprintf(stderr,"%s: protocol error, slave_z but no -gzip\n",__progname);
    exit(1);
  }
 switch (getchar())
  { case 'i':
	{ int mode;
	  int uid;
	  int gid;
	  int fd;
	  int n;
	  int r;
	  int left;
	  int err;
	  int rechmod;
	  unsigned long int from;
	  unsigned long int to;
	  sgetnts();
	  slavepathcheck();
	  sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
	  sgetnts();
	  sscanf(&slavework[0],"%d%d%d%lu%lu",&mode,&uid,&gid,&from,&to);
	  if (flag_debug) fprintf(stderr,"slave_z i %s %o %d %d %lu %lu\n",&slavetemp[0],mode&07777,uid,gid,from,to);
	  if (flag_readonly)
	   { printf("%d",EROFS);
	     putchar('\0');
	     return;
	   }
	  fd = open_to_update(&slavetemp[0],mode,&rechmod);
	  if (fd < 0)
	   { printf("%d",errno);
	     putchar('\0');
	     return;
	   }
	  if ( ( !flag_noown &&
		 ( (fchown(fd,-1,gid) < 0) ||
		   (fchown(fd,uid,-1) < 0) ) ) ||
	       (lseek(fd,from,SEEK_SET) < 0) )
	   { printf("%d",errno);
	     putchar('\0');
	     if (rechmod >= 0) fchmod(fd,rechmod);
	     close(fd);
	     return;
	   }
	  putchar('0');
	  putchar('\0');
	  fflush(stdout);
	  if (! flag_nosparse) fork_filter(-1,fd,0,sparsewrite);
	  fork_filter(-1,fd,"gunzip",(char *)0);
	  err = 0;
	  while (1)
	   { sgetnts();
	     sscanf(&slavework[0],"%d",&left);
	     if (left < 1) break;
	     while (left > 0)
	      { n = (left < WORKSIZE) ? left : WORKSIZE;
		r = fread(&slavework[0],1,n,stdin);
		if (r != n) exit(0); /* what else? */
		left -= r;
		if (err == 0)
		 { n = write(fd,&slavework[0],r);
		   if ((n >= 0) && (n < r))
		    { err = EIO;
		    }
		   else if (n < 0)
		    { err = errno;
		    }
		 }
	      }
	   }
	  if (rechmod >= 0) fchmod(fd,rechmod);
	  close(fd);
	  printf("%d",err);
	  putchar('\0');
	}
       break;
    case 'r':
	{ int fd;
	  int n;
	  sgetnts();
	  slavepathcheck();
	  sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
	  if (flag_debug) fprintf(stderr,"slave_z r %s\n",&slavetemp[0]);
	  fd = open(&slavetemp[0],O_RDONLY,0);
	  if (fd < 0)
	   { n = errno;
	     printf("-1");
	     putchar('\0');
	     printf("%d",errno);
	     putchar('\0');
	   }
	  else
	   { putchar('0');
	     putchar('\0');
	     fork_filter(fd,-1,"gzip",gzip_arg,(char *)0);
	     while (1)
	      { n = read(fd,&slavework[0],WORKSIZE);
		if (n < 0)
		 { n = errno;
		   printf("-1");
		   putchar('\0');
		   printf("%d",n);
		   putchar('\0');
		   break;
		 }
		printf("%d",n);
		putchar('\0');
		if (n < 1) break;
		fwrite(&slavework[0],1,n,stdout);
	      }
	     close(fd);
	   }
	}
       break;
  }
}

#if 0

static void slave_S(void)
{
 int fd;
 int cmd;
 int nb;
 int nr;
 long int off;
 long int curoff;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 fd = open(&slavetemp[0],O_RDONLY,0);
 if (flag_debug) fprintf(stderr,"slave_S entering command loop\n");
 curoff = 0;
 while (1)
  { fflush(stdout);
    cmd = getchar();
    switch (cmd)
     { default:
	  fprintf(stderr,"%s: protocol error, unknown slave S subcommand %d (%c)\n",__progname,cmd,cmd);
	  exit(1);
	  break;
       case EOF:
	  fprintf(stderr,"%s: protocol error, unexpected EOF (slave_S)\n",__progname);
	  exit(1);
	  break;
       case 'c':
	  goto out; /* aka break 2 */
	  break;
       case 'r':
	  sgetnts();
	  sscanf(&slavework[0],"%d%ld",&nb,&off);
	  if (nb > WORKSIZE)
	   { fprintf(stderr,"%s: protocol error, nb too big (%d > %d) in slave R r\n",__progname,nb,WORKSIZE);
	     exit(1);
	   }
	  if (off != curoff) lseek(fd,curoff,L_SET);
	  curoff = off;
	  nr = read(fd,&slavetemp[0],nb);
	  if (nr <= 0)
	   { printf("%d",COPYBLK_HOLE);
	     putchar('\0');
	   }
	  else
	   { if (nr < nb) bzero(&slavetemp[nr],nb-nr);
	     if (bcmp(&slavetemp[0],&zeroblk[0],nb))
	      { printf("%d",COPYBLK_DATA);
		putchar('\0');
		fwrite(&slavetemp[0],1,nb,stdout);
	      }
	     else
	      { printf("%d",COPYBLK_HOLE);
		putchar('\0');
	      }
	   }
	  curoff += (nr >= 0) ? nr : 0;
	  break;
     }
  }
out:;
 if (fd >= 0) close(fd);
}

static void slave_c(void)
{
 int mode;
 int uid;
 int gid;
 int size;
 int fd;
 int left;
 int blkleft;
 int blkoff;
 int flg;
 int n;
 int r;
 int tempfill;
 int skip;
 int w0;
 int writearg;
 int writerv;
 int finalerr;

 sgetnts();
 slavepathcheck();
 sprintf(&slavetemp[0],"%s/%s",&slavebase[0],&slavework[0]);
 sgetnts();
 sscanf(&slavework[0],"%d%d%d%d",&mode,&uid,&gid,&size);
 if (flag_debug) fprintf(stderr,"slave_c %s %o %d %d %d\n",&slavetemp[0],mode&07777,uid,gid,size);
 if (flag_readonly)
  { printf("%d",EROFS);
    putchar('\0');
    return;
  }
 fd = open(&slavetemp[0],O_WRONLY|O_CREAT|O_TRUNC,mode&07777);
 if (fd < 0)
  { printf("%d",errno);
    putchar('\0');
    return;
  }
 if (!flag_noown && ((fchown(fd,-1,gid) < 0) || (fchown(fd,uid,-1) < 0)))
  { printf("%d",errno);
    putchar('\0');
    close(fd);
    return;
  }
 putchar('0');
 putchar('\0');
 fflush(stdout);
 blkleft = 0;
 blkoff = 0;
 tempfill = 0;
 left = size;
 skip = 0;
 finalerr = 0;
#define WRITE(buf,nb) do { if ((writearg=(nb)) && (writerv=write(fd,(buf),writearg)) != writearg) goto writeerr; } while (0)
 while ((left > 0) || (blkleft > 0))
  { if (0)
     {
writeerr:;
       finalerr = (writerv < 0) ? errno : EIO;
       close(fd);
       fd = -1;
     }
    if (0)
     {
fderr:;
       finalerr = errno;
       close(fd);
       fd = -1;
     }
    if (blkleft < 1)
     { sgetnts();
       if (sscanf(&slavework[0],"%d%d",&n,&flg) != 2)
	{ fprintf(stderr,"%s: protocol error (can't read block size and type)\n",__progname);
	  exit(1);
	}
       blkleft += n;
       blkoff = 0;
       left -= n;
     }
    switch (flg)
     { case COPYBLK_DATA:
	  n = WORKSIZE - tempfill;
	  if (blkleft < n) n = blkleft;
	  r = fread(&slavework[tempfill],1,n,stdin);
	  if (r != n)
	   { fprintf(stderr,"%s: protocol error (can't read block data)\n",__progname);
	     exit(1);
	   }
	  blkleft -= n;
	  blkoff += n;
	  if (fd < 0) continue;
	  n += tempfill;
	  r = 0;
	  w0 = -1;
	  while (n-r >= 512)
	   { if (flag_nosparse || bcmp(&slavework[r],&zeroblk[0],512))
	      { if (w0 < 0) w0 = r;
		if (skip > 0)
		 { if (lseek(fd,skip,L_INCR) < 0) goto fderr;
		   skip = 0;
		 }
	      }
	     else
	      { if (w0 >= 0)
		 { WRITE(&slavework[w0],r-w0);
		   w0 = -1;
		 }
		skip += 512;
	      }
	     r += 512;
	   }
	  if (w0 >= 0) WRITE(&slavework[w0],r-w0);
	  tempfill = n - r;
	  if (tempfill && (r > 0)) bcopy(&slavework[r],&slavework[0],tempfill);
	  break;
       case COPYBLK_HOLE:
	  n = blkleft;
	  blkleft = 0;
	  blkoff += n;
	  if (fd < 0) continue;
	  if (tempfill+n < 512)
	   { bzero(&slavework[tempfill],n);
	     tempfill += n;
	   }
	  else
	   { if (flag_nosparse)
	      { if (tempfill > 0)
		 { bzero(&slavework[tempfill],512-tempfill);
		   WRITE(&slavework[0],512);
		   n -= 512 - tempfill;
		 }
		while (n >= WORKSIZE)
		 { WRITE(&zeroblk[0],WORKSIZE);
		   n -= WORKSIZE;
		 }
		if (n >= 512)
		 { WRITE(&zeroblk[0],n&~511);
		   n &= 511;
		 }
		if (n > 0) bzero(&slavework[0],n);
		tempfill = n;
	      }
	     else
	      { if (tempfill)
		 { if (bcmp(&slavework[0],&zeroblk[0],tempfill))
		    { bzero(&slavework[tempfill],512-tempfill);
		      if (skip > 0)
		       { if (lseek(fd,skip,L_INCR) < 0) goto fderr;
			 skip = 0;
		       }
		      WRITE(&slavework[0],512);
		      n -= 512 - tempfill;
		    }
		   else
		    { n += tempfill;
		    }
		 }
		skip += n;
		tempfill = skip % 512;
		if (tempfill)
		 { skip -= tempfill;
		   bzero(&slavework[0],tempfill);
		 }
	      }
	   }
	  break;
     }
  }
 if (tempfill > 0)
  { if (skip > 0)
     { if (lseek(fd,skip,L_INCR) < 0)
	{ finalerr = errno;
	  close(fd);
	  fd = -1;
	}
       skip = 0;
     }
    if (fd >= 0) WRITE(&slavework[0],tempfill);
  }
 if (fd >= 0)
  { setsize(fd,size);
    close(fd);
    putchar('0');
  }
 else
  { printf("%d",finalerr);
  }
#undef WRITE
 putchar('\0');
}

#endif

static void slavealarm(int sig __attribute__((__unused__)))
{
 exit(1);
}

static void slave(void)
{
 int cmd;

/*
 if (flag_trace) tracefilter(0,1,"slave");
 if (flag_encode) xfilter(0,1,"slave");
*/
 printf("V%d",CURRENT_VERSION);
 putchar('\0');
 fflush(stdout);
 sgetnts();
 if (flag_limitslave)
  { if (slavework[0] == '/')
     { fprintf(stderr,"%s: rooted path %s from master\n",__progname,&slavework[0]);
       exit(1);
     }
    slavepathcheck();
  }
 strcpy(&slavebase[0],force_dir?:&slavework[0]);
 signal(SIGALRM,slavealarm);
 umask(0);
 while (1)
  { fflush(stdout);
    alarm(60*60*12); /* twelve hours of idle time -> die, master probably crashed */
    cmd = getchar();
    switch (cmd)
     { default:
	  fprintf(stderr,"%s: protocol error, unknown slave command %d (%c)\n",__progname,cmd,cmd);
	  exit(1);
	  break;
       case EOF:
	  exit(0);
	  break;
       case 'D': slave_D(); break;
       case 'I': slave_I(); break;
       case 'K': slave_K(); break;
       case 'R': slave_R(); break;
       case 'd': slave_d(); break;
       case 'f': slave_f(); break;
       case 'i': slave_i(); break;
       case 'l': slave_l(); break;
       case 'm': slave_m(); break;
       case 'n': slave_n(); break;
       case 'o': slave_o(); break;
       case 'r': slave_r(); break;
       case 't': slave_t(); break;
       case 'u': slave_u(); break;
       case 'z': slave_z(); break;
/*
       case 'S': slave_S(); break;
       case 'c': slave_c(); break;
*/
     }
  }
}

static void set_nbio(int fd, int enb)
{
 ioctl(fd,FIONBIO,&enb);
}

static void finish_accept(REMOTE *r)
{
 int s;
 struct sockaddr_in sin;
 int sinsize;

 r->accepting = 0;
 sinsize = sizeof(sin);
 s = accept(r->rfd,(struct sockaddr *)&sin,&sinsize);
 if (s < 0)
  { fprintf(stderr,"%s: accept: %s\n",__progname,strerror(errno));
    r->errored = 1;
    return;
  }
 close(r->rfd);
 close(r->wfd);
 r->rfd = s;
 r->wfd = dup(s);
}

static void finish_connect(REMOTE *r)
{
 int e;
 int elen;

 r->connecting = 0;
 elen = sizeof(e);
 if (getsockopt(r->rfd,SOL_SOCKET,SO_ERROR,&e,&elen) < 0)
  { fprintf(stderr,"%s: getsockopt SO_ERROR: %s\n",__progname,strerror(errno));
    exit(1);
  }
 if (e == 0)
  { set_nbio(r->rfd,0);
    return;
  }
 fprintf(stderr,"%s: connect: %s\n",__progname,strerror(e));
 r->errored = 1;
}

static void await_netconns(void)
{
 REMOTE *r;
 fd_set rfds;
 fd_set wfds;
 int n;

 while (1)
  { FD_ZERO(&rfds);
    FD_ZERO(&wfds);
    n = 0;
    for (EACH_REM(r))
     { if (r->accepting)
	{ FD_SET(r->rfd,&rfds);
	  n ++;
	}
       if (r->connecting)
	{ FD_SET(r->rfd,&wfds);
	  n ++;
	}
     }
    if (n == 0) break;
    if (select(FD_SETSIZE,&rfds,&wfds,0,0) < 0)
     { if (errno == EINTR) continue;
       fprintf(stderr,"%s: select: %s\n",__progname,strerror(errno));
       exit(1);
     }
    for (EACH_REM(r))
     { if (r->accepting && FD_ISSET(r->rfd,&rfds)) finish_accept(r);
       if (r->connecting && FD_ISSET(r->rfd,&wfds)) finish_connect(r);
     }
  }
}

static void maybe_background(void)
{
 if (flag_bg)
  { int p;
    p = fork();
    if (p < 0)
     { fprintf(stderr,"%s: fork: %s\n",__progname,strerror(errno));
       exit(1);
     }
    if (p != 0) exit(0);
  }
 flag_bg = 0;
}

static int setup_connection_accept(REMOTE *r, unsigned short int nport, unsigned long int peer_a, unsigned long int peer_m)
{
 struct sockaddr_in sin;
 int a;

 bzero(&sin,sizeof(sin));
 sin.sin_family = AF_INET;
 sin.sin_addr.s_addr = INADDR_ANY;
 sin.sin_port = nport;
 a = socket(AF_INET,SOCK_STREAM,0);
 if (a < 0)
  { fprintf(stderr,"%s: socket: %s\n",__progname,strerror(errno));
    if (! r) exit(1);
    r->errored = 1;
    return(0);
  }
 if (bind(a,(struct sockaddr *)&sin,sizeof(sin)) < 0)
  { fprintf(stderr,"%s: bind: %s\n",__progname,strerror(errno));
    if (! r) exit(1);
    r->errored = 1;
    return(0);
  }
 if (listen(a,10) < 0)
  { fprintf(stderr,"%s: listen: %s\n",__progname,strerror(errno));
    if (! r) exit(1);
    r->errored = 1;
    return(0);
  }
 maybe_background();
 if (! r)
  { int s;
    struct sockaddr_in peer;
    int peersize;
    while (1)
     { peersize = sizeof(peer);
       s = accept(a,(struct sockaddr *)&peer,&peersize);
       if (s < 0)
	{ fprintf(stderr,"%s: accept: %s\n",__progname,strerror(errno));
	  exit(1);
	}
       if ((ntohl(peer.sin_addr.s_addr) & peer_m) == peer_a) break;
       close(s);
     }
    close(a);
    return(s);
  }
 r->accepting = 1;
 r->rfd = a;
 return(0);
}

static int setup_connection_connect(REMOTE *r, struct in_addr addr, unsigned short int nport)
{
 struct sockaddr_in sin;
 int s;

 bzero(&sin,sizeof(sin));
 sin.sin_family = AF_INET;
 sin.sin_addr = addr;
 sin.sin_port = nport;
 s = socket(AF_INET,SOCK_STREAM,0);
 if (s < 0)
  { fprintf(stderr,"%s: socket: %s\n",__progname,strerror(errno));
    if (! r) exit(1);
    r->errored = 1;
    return(0);
  }
 if (r) set_nbio(s,1);
 if (connect(s,(struct sockaddr *)&sin,sizeof(sin)) < 0)
  { if (errno == EINPROGRESS) /* "can't happen" if !r */
     { r->rfd = s;
       r->connecting = 1;
       return(0);
     }
    fprintf(stderr,"%s: connect: %s\n",__progname,strerror(errno));
    if (! r) exit(1);
    r->errored = 1;
    return(0);
  }
 if (! r) return(s);
 set_nbio(s,0);
 r->rfd = s;
 return(0);
}

static void connectremote(REMOTE *r)
{
 int i;
 int p1[2];
 int p2[2];
 const char *newav[64];
 const char **newavp;

 r->accepting = 0;
 r->connecting = 0;
 r->errored = 0;
 if (r->machine && !r->machine[0])
  { if (!strncmp(r->directory,"accept:",7))
     { unsigned short int p;
       char *cp;
       sscanf(r->directory,"accept:%hu",&p);
       cp = index(r->directory+7,':');
       if (cp == 0)
	{ fprintf(stderr,"%s: invalid :accept: remote spec\n",__progname);
	  exit(1);
	}
       r->directory = cp + 1;
       setup_connection_accept(r,htons(p),0,0);
     }
    else if (!strncmp(r->directory,"connect:",8))
     { unsigned short int p;
       char dqs[64];
       struct in_addr a;
       char *cp;
       sscanf(r->directory,"connect:%63[^:]:%hu",&dqs[0],&p);
       dqs[63] = '\0';
       a.s_addr = inet_addr(&dqs[0]);
       cp = index(r->directory+8,':');
       if (cp == 0)
	{ fprintf(stderr,"%s: invalid :connect: remote spec\n",__progname);
	  exit(1);
	}
       cp = index(cp+1,':');
       if (cp == 0)
	{ fprintf(stderr,"%s: invalid :connect: remote spec\n",__progname);
	  exit(1);
	}
       r->directory = cp + 1;
       setup_connection_connect(r,a,htons(p));
     }
    else
     { fprintf(stderr,"%s: invalid keyword-style remote spec: %s\n",__progname,r->directory);
       exit(1);
     }
    r->wfd = dup(r->rfd);
    if (r->wfd < 0)
     { fprintf(stderr,"%s: dup: %s\n",__progname,strerror(errno));
       exit(1);
     }
    return;
  }
 if ((pipe(&p1[0]) < 0) || (pipe(&p2[0]) < 0))
  { fprintf(stderr,"%s: can't create pipe: %s\n",__progname,strerror(errno));
    exit(1);
  }
 switch (fork())
  { case -1:
       fprintf(stderr,"%s: can't fork: %s\n",__progname,strerror(errno));
       exit(1);
       break;
    case 0:
       break;
    default:
       r->wfd = p1[1];
       close(p1[0]);
       r->rfd = p2[0];
       close(p2[1]);
       return;
       break;
  }
 close(p1[1]);
 close(p2[0]);
 if (p1[0] != 0)
  { dup2(p1[0],0);
    close(p1[0]);
  }
 if (p2[1] != 1)
  { dup2(p2[1],1);
    close(p2[1]);
  }
 for (i=getdtablesize()-1;i>2;i--) close(i);
 if (! r->machine)
  { slave();
    exit(0);
  }
 newavp = &newav[0];
 *newavp++ = r->rsh;
 *newavp++ = r->machine;
 if (r->user)
  { *newavp++ = "-l";
    *newavp++ = r->user;
  }
 *newavp++ = r->program ? r->program : defprogram;
 *newavp++ = "-R";
 if (flag_debug) *newavp++ = "-debug";
 if (flag_follow) *newavp++ = "-follow";
 if (flag_sgiddir) *newavp++ = "-sgid-dirs";
 if (flag_noown) *newavp++ = "-no-owners";
 if (flag_mtimes) *newavp++ = "-mtimes";
 if (flag_encode) *newavp++ = "-encode";
 if (flag_trace) *newavp++ = "-trace";
 if (flag_nosparse) *newavp++ = "-no-sparse";
 if (flag_links) *newavp++ = "-links";
 if (flag_gzip)
  { *newavp++ = "-gzip";
    *newavp++ = gzip_arg;
  }
 *newavp++ = 0;
 execvp(r->rsh,(const void *)&newav[0]);
 printf("F%s",strerror(errno));
 putchar('\0');
 exit(0);
}

static void setupremote(REMOTE *r)
{
 if (flag_trace) tracefilter(r->rfd,r->wfd,r->fullarg);
 if (flag_encode) xfilter(r->rfd,r->wfd,r->fullarg);
 r->rf = fdopen(r->rfd,"r");
 r->wf = fdopen(r->wfd,"w");
}

static void runremotes(void)
{
 REMOTE *r;

 signal(SIGCHLD,reaper);
 for (EACH_REM(r)) connectremote(r);
 await_netconns();
 for (EACH_REM(r)) if (r->errored) exit(1);
 for (EACH_REM(r)) setupremote(r);
}

static void mgetnts(REMOTE *r)
{
 int x;
 int c;

 x = 0;
 while (1)
  { c = getc(r->rf);
    if (c == EOF)
     { fprintf(stderr,"%s: unexpected eof (or error) reading from %s\n",__progname,r->fullarg);
       exit(1);
     }
    r->work[x] = c;
    if (c == 0)
     { r->worklen = x;
       return;
     }
    if (x < WORKSIZE-1) x ++;
  }
}

static void startup_remotes(void)
{
 REMOTE *r;
 int errs;
 int vers;

 errs = 0;
 for (EACH_REM(r))
  { int code;
    code = getc(r->rf);
    if (code == 'V')
     { mgetnts(r);
       if (sscanf(&r->work[0],"%d",&vers) != 1)
	{ fprintf(stderr,"%s: protocol error 1 starting remote for %s\n",__progname,r->fullarg);
	}
       else if (vers != CURRENT_VERSION)
	{ fprintf(stderr,"%s: protocol version mismatch with remote for %s\n",__progname,r->fullarg);
	}
       else
	{ fputs(r->directory,r->wf);
	  putc('\0',r->wf);
	  continue;
	}
     }
    errs ++;
    if (code == 'F')
     { mgetnts(r);
       fprintf(stderr,"%s: can't start remote for %s: %s\n",__progname,r->fullarg,&r->work[0]);
     }
    else
     { fprintf(stderr,"%s: protocol error 2 starting remote for %s%s\n",__progname,r->fullarg,r->machine?" (permission problem?)":"");
     }
  }
 if (errs) exit(1);
}

/* no, this is not a proper polyphase merge sort! */
static ENTRY *entsort(ENTRY *el)
{
 ENTRY_TCONC tape1;
 ENTRY_TCONC tape2;
 ENTRY_TCONC tape3;
 ENTRY_TCONC *tapea;
 ENTRY_TCONC *tapeb;
 const char *last;
 ENTRY *e;
 ENTRY *f;
 int fok;

 if ((el == 0) || (el->link == 0)) return(el);
 tape3.list = el;
 while (1)
  { tapea = &tape1;
    tapeb = &tape2;
    etc_init(tapea);
    etc_init(tapeb);
    last = "";
    for (e=tape3.list;e;e=e->link)
     { if (strcmp(last,e->name) > 0)
	{ ENTRY_TCONC *tapet;
	  tapet = tapea;
	  tapea = tapeb;
	  tapeb = tapet;
	}
       etc_add(tapea,e);
       last = e->name;
     }
    etc_end(tapea);
    etc_end(tapeb);
    if (tapeb->list == 0) return(tapea->list);
    etc_init(&tape3);
    e = tape1.list;
    f = tape2.list;
    last = 0;
    while (e || f)
     { fok = (f && (!last || (strcmp(last,f->name) <= 0)));
       if ( e &&
	    (!last || (strcmp(last,e->name) <= 0)) &&
	    (!fok || (strcmp(e->name,f->name) <= 0)) )
	{ etc_add(&tape3,e);
	  last = e->name;
	  e = e->link;
	}
       else if (fok)
	{ ENTRY *t;
	  t = e;
	  e = f;
	  f = t;
	}
       else
	{ last = 0;
	}
     }
  }
}

static void get_dirlist(REMOTE *r, const char *dir)
{
 ENTRY *e;
 int maj;
 int min;
 char *wp;

 r->dirlist = 0;
 fprintf(r->wf,"D%s",dir);
 putc('\0',r->wf);
 fflush(r->wf);
 while (1)
  { mgetnts(r);
    if (r->worklen == 0) break;
    e = newentry();
    e->link = r->dirlist;
    r->dirlist = e;
    e->name = copyofstr(&r->work[0]);
    mgetnts(r);
    wp = &r->work[0];
    e->mode = strtol(wp,&wp,0);
    e->uid = strtol(wp,&wp,0);
    e->gid = strtol(wp,&wp,0);
    if (flag_mtimes) e->mtime = strtol(wp,&wp,0);
    if (flag_links) e->nlinks = strtol(wp,&wp,0);
    switch (e->mode & S_IFMT)
     { case S_IFDIR:
       case S_IFSOCK:
	  break;
       case S_IFCHR:
       case S_IFBLK:
	  maj = strtol(wp,&wp,0);
	  min = strtol(wp,&wp,0);
	  e->rdev = makedev(maj,min);
	  break;
       case S_IFREG:
	  e->size = strtoul(wp,&wp,0);
	  break;
       case S_IFLNK:
	  mgetnts(r);
	  e->linkto = copyofstr(&r->work[0]);
	  break;
     }
  }
}

static void installit(MASK, const char *, MASK *); /* forward */

static void rmcpr(MASK m, const char *dir, char *file, MASK *failmask)
{
 MASK *fm;
 REMOTE *r;
 ENTRY *e;
 int err;
 char *addl;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmcpr mask includes umaster");
 if (strcmp(file,e->name)) panic("rmcpr file name doesn't match entry name");
  { static MASK fail;
    fm = failmask ? failmask : &fail;
  }
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { fprintf(r->wf,"d%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d %d %d",e->mode,e->uid,e->gid);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ printf("      mkdir of %s/%s on %s failed: %s\n",dir,file,r->fullarg,strerror(err));
	  MASK_CLR(m,r->bit);
	  MASK_SET(*fm,r->bit);
	}
     }
  }
 if (! MASK_ISZERO(m))
  { addl = malloc(strlen(dir)+1+strlen(file)+1);
    sprintf(addl,"%s/%s",dir,file);
    get_dirlist(umaster,addl);
    if (umaster->dirlist)
     { while (umaster->dirlist)
	{ ENTRY *et;
	  et = umaster->dirlist;
	  et->mode = (et->mode & perm_and) | perm_or;
	  installit(m,addl,fm);
	  umaster->dirlist = et->link;
	  freeentry(et);
	}
     }
    free(addl);
    if (flag_mtimes)
     { for (EACH_REM(r))
	{ if (MASK_TST(m,r->bit))
	   { fprintf(r->wf,"u%s/%s",dir,file);
	     putc('\0',r->wf);
	     fprintf(r->wf,"%ld 0",e->mtime);
	     putc('\0',r->wf);
	   }
	}
     }
  }
 umaster->dirlist = e;
 if (! failmask)
  { for (EACH_REM(r)) if (MASK_TST(m,r->bit) && !MASK_TST(*fm,r->bit)) printf("      directory copy to %s succeeded\n",r->fullarg);
  }
}

static void rmmknod(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmmknod mask includes umaster");
 if (strcmp(file,e->name)) panic("rmmknod file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"n%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d %d %d %d",e->mode,e->uid,e->gid,e->rdev);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      mknod on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      mknod on %s succeeded\n",r->fullarg);
	}
     }
  }
}

static void rmsymlink(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmsymlink mask includes umaster");
 if (strcmp(file,e->name)) panic("rmsymlink file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"l%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d %d %d",umaster->dirlist->mode,umaster->dirlist->uid,umaster->dirlist->gid);
       putc('\0',r->wf);
       fprintf(r->wf,"%s",e->linkto);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      symlink on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      symlink on %s succeeded\n",r->fullarg);
	}
     }
  }
}

#ifdef S_IFIFO
static void rmmkfifo(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmmkfifo mask includes umaster");
 if (strcmp(file,e->name)) panic("rmmkfifo file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"f%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d %d %d",e->mode,e->uid,e->gid);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      mkfifo on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      mkfifo on %s succeeded\n",r->fullarg);
	}
     }
  }
}
#endif

static void rmutimes(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmutimes mask includes umaster");
 if (strcmp(file,e->name)) panic("rmutimes file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"u%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%ld 0",e->mtime);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      utimes on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      utimes on %s succeeded\n",r->fullarg);
	}
     }
  }
}

static void rmcp(MASK m, const char *dir, const char *file, MASK *failmask, unsigned long int fromoff, unsigned long int tooff)
{
 REMOTE *r;
 ENTRY *e;
 unsigned long int left;
 int n;
 unsigned long int off;
 int err;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmcp mask includes umaster");
 if (strcmp(file,e->name)) panic("rmcp file name doesn't match entry name");
 if (fromoff > e->size) panic("rmcp begins after eof");
 if (tooff > e->size) tooff = e->size;
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { if (flag_gzip)
	{ fprintf(r->wf,"zi%s/%s",dir,file);
	  putc('\0',r->wf);
	  fprintf(r->wf,"%d %d %d %lu %lu",e->mode,e->uid,e->gid,fromoff,tooff);
	  putc('\0',r->wf);
	}
       else
	{ fprintf(r->wf,"i%s/%s",dir,file);
	  putc('\0',r->wf);
	  fprintf(r->wf,"%d %d %d %lu %lu %lu",e->mode,e->uid,e->gid,e->size,fromoff,tooff);
	  putc('\0',r->wf);
	}
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err != 0)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      copy to %s failed: %s\n",r->fullarg,strerror(err));
	   }
	  MASK_CLR(m,r->bit);
	}
     }
  }
 if (! MASK_ISZERO(m))
  { if (flag_gzip)
     { fprintf(umaster->wf,"zI%s/%s",dir,file);
       putc('\0',umaster->wf);
       fprintf(umaster->wf,"%lu %lu",fromoff,tooff);
       putc('\0',umaster->wf);
       fflush(umaster->wf);
       mgetnts(umaster);
       sscanf(&umaster->work[0],"%d",&n);
       if (n < 0)
	{ mgetnts(umaster);
	  sscanf(&umaster->work[0],"%d",&err);
	  if (failmask)
	   { mask_setor(failmask,m);
	   }
	  else
	   { printf("      open %s failed: %s\n",umaster->fullarg,strerror(err));
	   }
	}
       else
	{ while (1)
	   { mgetnts(umaster);
	     sscanf(&umaster->work[0],"%d",&n);
	     for (EACH_REM(r)) if (MASK_TST(m,r->bit))
	      { fprintf(r->wf,"%d",n);
		putc('\0',r->wf);
	      }
	     if (n < 1) break;
	     left = n;
	     while (left > 0)
	      { n = (left < WORKSIZE) ? left : WORKSIZE;
		fread(&umaster->work[0],1,n,umaster->rf);
		for (EACH_REM(r)) if (MASK_TST(m,r->bit)) fwrite(&umaster->work[0],1,n,r->wf);
		left -= n;
	      }
	   }
	  if (n < 0)
	   { mgetnts(umaster);
	     sscanf(&umaster->work[0],"%d",&err);
	     if (failmask)
	      { mask_setor(failmask,m);
	      }
	     else
	      { printf("      reading from %s failed: %s\n",umaster->fullarg,strerror(err));
	      }
	   }
	}
     }
    else
     { fprintf(umaster->wf,"I%s/%s",dir,file);
       putc('\0',umaster->wf);
       fflush(umaster->wf);
       left = tooff - fromoff;
       off = fromoff;
       while (left > 0)
	{ int flg;
	  n = (left < WORKSIZE) ? left : WORKSIZE;
	  fprintf(umaster->wf,"r%d %lu",n,off);
	  putc('\0',umaster->wf);
	  fflush(umaster->wf);
	  mgetnts(umaster);
	  if (sscanf(&umaster->work[0],"%d",&flg) != 1)
	   { fprintf(stderr,"%s: protocol error: update master returned no flag\n",__progname);
	   }
	  switch (flg)
	   { case COPYBLK_DATA:
		fread(&umaster->work[0],1,n,umaster->rf);
		for (EACH_REM(r))
		 { if (MASK_TST(m,r->bit))
		    { fprintf(r->wf,"%d %d",n,COPYBLK_DATA);
		      putc('\0',r->wf);
		      fwrite(&umaster->work[0],1,n,r->wf);
		    }
		 }
		break;
	     case COPYBLK_HOLE:
		for (EACH_REM(r))
		 { if (MASK_TST(m,r->bit))
		    { fprintf(r->wf,"%d %d",n,COPYBLK_HOLE);
		      putc('\0',r->wf);
		    }
		 }
		break;
	     default:
		fprintf(stderr,"%s: update master returned unknown block code %d\n",__progname,flg);
		exit(1);
		break;
	   }
	  off += n;
	  left -= n;
	}
       putc('c',umaster->wf);
     }
    for (EACH_REM(r))
     { if (MASK_TST(m,r->bit))
	{ fflush(r->wf);
	  mgetnts(r);
	  sscanf(&r->work[0],"%d",&err);
	  if (err)
	   { if (failmask)
	      { MASK_SET(*failmask,r->bit);
	      }
	     else
	      { printf("      copy to %s failed: %s\n",r->fullarg,strerror(err));
	      }
	   }
	  else
	   { if (! failmask) printf("      copy to %s succeeded\n",r->fullarg);
	   }
	}
     }
  }
}

static void installreg(MASK mask, const char *addl, const char *name, MASK *failmask, unsigned long int fromoff, unsigned long int tooff)
{
 rmcp(mask,addl,name,failmask,fromoff,tooff);
}

static void installit(MASK mask, const char *addl, MASK *failmask)
{
 ENTRY *e;

 e = umaster->dirlist;
 switch (e->mode & S_IFMT)
  { default:
       if (failmask)
	{ *failmask = mask;
	}
       else
	{ printf("      install failed: unknown mode %o for %s in %s:%s\n",e->mode,e->name,umaster->fullarg,addl);
	}
       break;
    case S_IFDIR:
       rmcpr(mask,addl,e->name,failmask);
       break;
    case S_IFCHR:
    case S_IFBLK:
       rmmknod(mask,addl,e->name,failmask);
       if (flag_mtimes) rmutimes(mask,addl,e->name,failmask);
       break;
    case S_IFREG:
       installreg(mask,addl,e->name,failmask,0,~0UL);
       if (flag_mtimes) rmutimes(mask,addl,e->name,failmask);
       break;
    case S_IFLNK:
       rmsymlink(mask,addl,e->name,failmask);
       break;
    case S_IFSOCK:
       if (failmask)
	{ *failmask = mask;
	}
       else
	{ printf("      install %s/%s failed: can't copy sockets\n",addl,e->name);
	}
       break;
#ifdef S_IFIFO
    case S_IFIFO:
       rmmkfifo(mask,addl,e->name,failmask);
       if (flag_mtimes) rmutimes(mask,addl,e->name,failmask);
       break;
#endif
  }
}

static int rmrm(REMOTE *r, const char *dir, char *name)
{
 int err;

 fprintf(r->wf,"r%s/%s",dir,name);
 putc('\0',r->wf);
 fflush(r->wf);
 mgetnts(r);
 sscanf(&r->work[0],"%d",&err);
 return(err);
}

static void rmtrunc(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmtrunc mask includes umaster");
 if (strcmp(file,e->name)) panic("rmtrunc file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"t%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%lu",e->size);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      truncate on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      truncate on %s succeeded\n",r->fullarg);
	}
     }
  }
}

static void rmchmod(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmchmod mask includes umaster");
 if (strcmp(file,e->name)) panic("rmchmod file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"m%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d",e->mode&07777);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      chmod on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      chmod on %s succeeded\n",r->fullarg);
	}
     }
  }
}

static void rmchown(MASK m, const char *dir, char *file, MASK *failmask)
{
 REMOTE *r;
 ENTRY *e;

 e = umaster->dirlist;
 if (MASK_TST(m,umaster->bit)) panic("rmchown mask includes umaster");
 if (strcmp(file,e->name)) panic("rmchown file name doesn't match entry name");
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { int err;
       fprintf(r->wf,"o%s/%s",dir,file);
       putc('\0',r->wf);
       fprintf(r->wf,"%d %d",e->uid,e->gid);
       putc('\0',r->wf);
       fflush(r->wf);
       mgetnts(r);
       sscanf(&r->work[0],"%d",&err);
       if (err)
	{ if (failmask)
	   { MASK_SET(*failmask,r->bit);
	   }
	  else
	   { printf("      chown on %s failed: %s\n",r->fullarg,strerror(err));
	   }
	}
       else
	{ if (! failmask) printf("      chown on %s succeeded\n",r->fullarg);
	}
     }
  }
}

static char *modestring(int mode)
{
 static char modebuf[64];

 switch (mode & S_IFMT)
  { case S_IFDIR:  modebuf[0] = 'd'; break;
    case S_IFCHR:  modebuf[0] = 'c'; break;
    case S_IFBLK:  modebuf[0] = 'b'; break;
    case S_IFREG:  modebuf[0] = '-'; break;
    case S_IFLNK:  modebuf[0] = 'l'; break;
    case S_IFSOCK: modebuf[0] = 's'; break;
#ifdef S_IFIFO
    case S_IFIFO:  modebuf[0] = 'p'; break;
#endif
    default:
       sprintf(&modebuf[0],"?%07o",mode&07777777);
       return(&modebuf[0]);
       break;
  }
 modebuf[1] = (mode & 0400) ? 'r' : '-';
 modebuf[2] = (mode & 0200) ? 'w' : '-';
 switch (mode & (S_ISUID|0100))
  { case 0           : modebuf[3] = '-'; break;
    case S_ISUID     : modebuf[3] = 'S'; break;
    case         0100: modebuf[3] = 'x'; break;
    case S_ISUID|0100: modebuf[3] = 's'; break;
  }
 modebuf[4] = (mode & 040) ? 'r' : '-';
 modebuf[5] = (mode & 020) ? 'w' : '-';
 switch (mode & (S_ISGID|010))
  { case 0          : modebuf[6] = '-'; break;
    case S_ISGID    : modebuf[6] = 'S'; break;
    case         010: modebuf[6] = 'x'; break;
    case S_ISGID|010: modebuf[6] = 's'; break;
  }
 modebuf[7] = (mode & 04) ? 'r' : '-';
 modebuf[8] = (mode & 02) ? 'w' : '-';
 switch (mode & (S_ISVTX|01))
  { case 0         : modebuf[9] = '-'; break;
    case S_ISVTX   : modebuf[9] = 'T'; break;
    case         01: modebuf[9] = 'x'; break;
    case S_ISVTX|01: modebuf[9] = 't'; break;
  }
 modebuf[10] = '\0';
 return(&modebuf[0]);
}

static void get_md5_sums(MASK m, const char *dir, const char *name, unsigned long int from, unsigned long int to)
{
 REMOTE *r;

 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { fprintf(r->wf,"K%s/%s",dir,name);
       putc('\0',r->wf);
       fprintf(r->wf,"%lu %lu",from,to);
       putc('\0',r->wf);
       fflush(r->wf);
     }
  }
 for (EACH_REM(r))
  { if (MASK_TST(m,r->bit))
     { fread(&r->work[0],1,1,r->rf);
       switch (r->work[0])
	{ case 'O': case 'R':
	     mgetnts(r);
	     r->work[0] = 0;
	     break;
	  case 'K':
	     fread(&r->work[1],1,16,r->rf);
	     r->work[0] = 1;
	     break;
	}
     }
  }
}

static MASK compare_reg_topmask;
static int compare_reg_firstset;
static int compare_reg_size;
static int compare_reg_psize;

static void showhdg(void)
{
 REMOTE *r;

 if (flag_check) exit(1);
 if (needhdg)
  { printf("%s/%s:\n",cur_dir,cur_ent);
    for (EACH_REM(r)) if (MASK_TST(cur_nx,r->bit)) printf("    nonexistent in %s\n",r->fullarg);
    needhdg = 0;
  }
}

static void compare_reg_set(MASK mask)
{
 REMOTE *r;

 if (flag_debug) fprintf(stderr,"compare_reg_set mask=0x%lx\n",mask.bits[0]);
 if (MASK_EQ(mask,compare_reg_topmask)) return;
 showhdg();
 if (compare_reg_firstset)
  { printf("    plain file %s vary",flag_md5?"checksums":"contents");
    if (compare_reg_psize) printf(" for size %d",compare_reg_size);
    printf(":\n");
  }
 else
  { printf("    --------\n");
  }
 compare_reg_firstset = 0;
 for (EACH_REM(r)) if (MASK_TST(mask,r->bit))
  { printf("\t%s",r->fullarg);
    if (flag_md5 && !r->work[0]) printf(" (open/read failure)");
    printf("\n");
  }
}

static void compare_reg_aux(MASK mask, long int off0)
{
 MASK todo;
 MASK these;
 REMOTE *r;
 REMOTE *cmpr;
 long int off;
 int n;

 if (MASK_ISZERO(mask)) panic("compare_reg_aux zero mask");
 if (MASK_ONEBIT(mask))
  { compare_reg_set(mask);
    return;
  }
 if (flag_debug)
  { fprintf(stderr,"compare_reg_aux starting off0=%ld mask=0x%lx\n",off0,mask.bits[0]);
    for (EACH_REM(r)) if (MASK_TST(mask,r->bit)) fprintf(stderr,"\t%s\n",r->fullarg);
  }
 off = off0;
 while (1)
  { if (flag_debug) fprintf(stderr,"compare_reg_aux top of loop, off=%ld\n",off);
    n = compare_reg_size - off;
    if (n > WORKSIZE) n = WORKSIZE;
    if (n < 1)
     { compare_reg_set(mask);
       return;
     }
    for (EACH_REM(r))
     { if (MASK_TST(mask,r->bit))
	{ fprintf(r->wf,"r%d %ld",n,off);
	  putc('\0',r->wf);
	  fflush(r->wf);
	  fread(&r->work[0],1,n,r->rf);
	}
     }
    off += n;
    if (flag_debug) fprintf(stderr,"compare_reg_aux off advanced to %ld\n",off);
    todo = mask;
    while (1)
     { for (EACH_REM(r)) if (MASK_TST(todo,r->bit)) break;
       cmpr = r;
       if (cmpr == 0)
	{ panic("compare_reg_aux can't find a remote to do");
	}
       MASK_ZERO(these);
       for (EACH_REM(r))
	{ if (MASK_TST(todo,r->bit))
	   { if ((r == cmpr) || !bcmp(&r->work[0],&cmpr->work[0],n))
	      { MASK_SET(these,r->bit);
		MASK_CLR(todo,r->bit);
	      }
	   }
	}
       if (MASK_ISZERO(todo)) break;
       compare_reg_aux(these,off);
     }
    mask = these;
  }
}

static void compare_reg(MASK mask, int size, int printsize)
{
 REMOTE *r;

 compare_reg_topmask = mask;
 compare_reg_size = size;
 compare_reg_psize = printsize;
 compare_reg_firstset = 1;
 if (MASK_ONEBIT(mask)) return;
 if (flag_md5)
  { MASK m;
    MASK left;
    REMOTE *r2;
    get_md5_sums(mask,cur_dir,cur_ent,0,size);
    left = mask;
    while (! MASK_ISZERO(left))
     { for (EACH_REM(r)) if (MASK_TST(left,r->bit)) break;
       MASK_ZERO(m);
       if (r->work[0])
	{ for (EACH_REM(r2)) if (MASK_TST(left,r2->bit) && r2->work[0] && !bcmp(&r->work[1],&r2->work[1],16)) MASK_SET(m,r2->bit);
	}
       else
	{ MASK_SET(m,r->bit);
	}
       compare_reg_set(m);
       mask_setandnot(&left,m);
     }
  }
 else
  { for (EACH_REM(r))
     { if (MASK_TST(mask,r->bit))
	{ fprintf(r->wf,"R%s/%s",cur_dir,cur_ent);
	  putc('\0',r->wf);
	  fflush(r->wf);
	}
     }
    compare_reg_aux(mask,0);
    for (EACH_REM(r)) if (MASK_TST(mask,r->bit)) putc('c',r->wf);
  }
}

static void check_reg_diff(REMOTE *mr, const char *dir, MASK *m, unsigned long int fromsize, unsigned long int tosize)
{
 int left;
 int n;
 unsigned long int off;
 REMOTE *r;
 MASK w;
 MASK all;

 if (MASK_TST(*m,mr->bit)) panic("check_reg_diff master in mask");
 if (fromsize >= tosize)
  { if ((fromsize == 0) && (tosize == 0))
     { MASK_ZERO(*m);
       return;
     }
    panic("check_reg_diff fromsize >= tosize");
  }
 if (tosize > mr->dirlist->size) panic("check_reg_diff tosize > master size");
 if (MASK_ISZERO(*m)) return;
 w = *m;
 all = w;
 if (flag_debug) fprintf(stderr,"check_reg_diff master is %s\n",mr->fullarg);
 MASK_ZERO(*m);
 for (EACH_REM(r))
  { if ( MASK_TST(w,r->bit) &&
	 (ulmin(r->dirlist->size,tosize) != ulmin(mr->dirlist->size,tosize)) )
     { if (flag_debug) fprintf(stderr,"check_reg_diff sees %s differs in size\n",r->fullarg);
       MASK_CLR(w,r->bit);
       MASK_SET(*m,r->bit);
     }
  }
 if (flag_md5)
  { MASK_SET(w,mr->bit);
    get_md5_sums(w,dir,mr->dirlist->name,fromsize,tosize);
    MASK_CLR(w,mr->bit);
    for (EACH_REM(r)) if (MASK_TST(w,r->bit) && (!mr->work[0] || !r->work[0] || bcmp(&mr->work[1],&r->work[1],16)))
     { if (flag_debug) fprintf(stderr,"check_reg_diff sees %s differs in checksum\n",r->fullarg);
       MASK_SET(*m,r->bit);
     }
  }
 else
  { for (EACH_REM(r))
     { if ((r == mr) || MASK_TST(w,r->bit))
	{ fprintf(r->wf,"R%s/%s",dir,mr->dirlist->name);
	  putc('\0',r->wf);
	  fflush(r->wf);
	}
     }
    left = tosize - fromsize;
    off = fromsize;
    while (1)
     { if (flag_debug) fprintf(stderr,"check_reg_diff top of loop, off=%ld\n",off);
       n = (left < WORKSIZE) ? left : WORKSIZE;
       if (n < 1)
	{ for (EACH_REM(r)) if ((r == mr) || MASK_TST(all,r->bit)) putc('c',r->wf);
	  return;
	}
       for (EACH_REM(r))
	{ if ((r == mr) || MASK_TST(w,r->bit))
	   { fprintf(r->wf,"r%d %ld",n,off);
	     putc('\0',r->wf);
	     fflush(r->wf);
	     fread(&r->work[0],1,n,r->rf);
	   }
	}
       off += n;
       left -= n;
       if (flag_debug) fprintf(stderr,"check_reg_diff off advanced to %ld\n",off);
       for (EACH_REM(r))
	{ if (MASK_TST(w,r->bit))
	   { if (bcmp(&mr->work[0],&r->work[0],n))
	      { if (flag_debug) fprintf(stderr,"check_reg_diff finds %s differs\n",r->fullarg);
		MASK_CLR(w,r->bit);
		MASK_SET(*m,r->bit);
	      }
	   }
	}
     }
  }
}

static int wildmatch(const char *pat, const char *s, const char *s1, const char *s2)
{
 while (1)
  { if (!*s && s1)
     { s = s1;
       s1 = s2;
       s2 = 0;
     }
    switch (*pat)
     { case '?':
	  if (! *s) return(0);
	  break;
       case '[':
	   { int found;
	     int neg;
	     int first;
	     const char *pp;
	     found = 0;
	     neg = 0;
	     first = 1;
	     pp = pat + 1;
	     if (*pp == '^')
	      { neg = 1;
		pp ++;
	      }
	     while (first || (*pp != ']'))
	      { if ( (pp[1] == '-') &&
		     pp[2] && (pp[2] != ']') &&
		     (pp[0] <= pp[2]) )
		 { if ((pp[0] <= *s) && (*s <= pp[2])) found = 1;
		   pp += 3;
		 }
		else
		 { if (pp[0] == *s) found = 1;
		   pp ++;
		 }
		first = 0;
	      }
	     if (! *pp)
	      { fprintf(stderr,"%s: unclosed [ in pattern\n",__progname);
		exit(1);
	      }
	     if (neg?found:!found) return(0);
	     pat = pp;
	   }
	  break;
       case '*':
	  if (wildmatch(pat+1,s,s1,s2)) return(1);
	  if (! *s) return(0);
	  s ++;
	  continue;
	  break;
       case '\0':
	  if (*s) return(0);
	  return(1);
	  break;
       default:
	  if (*pat != *s) return(0);
	  break;
     }
    pat ++;
    s ++;
  }
}

static int prunematch(PRUNE *p, const char *dir, const char *ent)
{
 switch (p->type)
  { case PT_PATH:
       if (dir)
	{ int dl;
	  dl = strlen(dir);
	  if ( !bcmp(dir,p->path,dl) &&
	       (p->path[dl] == '/') &&
	       !strcmp(p->path+dl+1,ent) ) return(1);
	}
       else
	{ if (!strcmp(p->path,ent)) return(1);
	}
       break;
    case PT_WILD:
       if (wildmatch(p->path,dir?dir:ent,dir?"/":0,dir?ent:0)) return(1);
       break;
  }
 return(0);
}

static void newfile_init(const char *dir, char *ent)
{
 PRUNE *p;
 PRUNE **pp;

 cur_dir = dir;
 cur_ent = ent;
 cur_pruned = 0;
 if ((dir[0] == '.') && !dir[1])
  { pp = &prunelist;
    while ((p = *pp))
     { if (prunematch(p,0,ent)) break;
       pp = &p->link;
     }
  }
 else
  { if ((dir[0] == '.') && (dir[1] == '/')) dir += 2;
    pp = &prunelist;
    while ((p = *pp))
     { if (prunematch(p,dir,ent)) break;
       pp = &p->link;
     }
  }
 if (p)
  { cur_pruned = 1;
    switch (p->type)
     { case PT_PATH:
	  *pp = p->link;
	  OLD(p);
	  break;
       case PT_WILD:
	  break;
     }
  }
 needhdg = 1;
 if (prunex) cur_pruned = ! cur_pruned;
}

static void endfile_flush(void)
{
 if (! needhdg) fflush(stdout);
}

#define ENTDIFF_TYPE 0x00000001
#define ENTDIFF_MODE 0x00000002
#define ENTDIFF_UID  0x00000004
#define ENTDIFF_GID  0x00000008
#define ENTDIFF_SIZE 0x00000010
#define ENTDIFF_RDEV 0x00000020
#define ENTDIFF_LINK 0x00000040
#define ENTDIFF_TIME 0x00000080
#define ENTDIFF_NLNK 0x00000100
static int entdiff(ENTRY *e1, ENTRY *e2)
{
 int how;

 if ((e1->mode & S_IFMT) != (e2->mode & S_IFMT)) return(ENTDIFF_TYPE);
 how = 0;
 if ((e1->mode ^ e2->mode) & perm_mask) how |= ENTDIFF_MODE;
 if (! flag_noown)
  { if (e1->uid != e2->uid) how |= ENTDIFF_UID;
    if (e1->gid != e2->gid) how |= ENTDIFF_GID;
  }
 if (flag_mtimes && (e1->mtime != e2->mtime)) how |= ENTDIFF_TIME;
 if (flag_links && (e1->nlinks != e2->nlinks)) how |= ENTDIFF_NLNK;
 switch (e1->mode & S_IFMT)
  { default:
       return(0);
       break;
    case S_IFDIR:
       if ( flag_sgiddir &&
	    (how & ENTDIFF_MODE) &&
	    ((e1->mode&05777&perm_mask) == (e2->mode&05777&perm_mask)) )
	{ how &= ~ENTDIFF_MODE;
	}
       break;
    case S_IFCHR:
    case S_IFBLK:
       if (e1->rdev != e2->rdev) how |= ENTDIFF_RDEV;
       break;
    case S_IFREG:
       if (e1->size != e2->size) how |= ENTDIFF_SIZE;
       break;
    case S_IFLNK:
       if (strcmp(e1->linkto,e2->linkto)) how |= ENTDIFF_LINK;
       how &= ~ENTDIFF_TIME;
       break;
    case S_IFSOCK:
       break;
#ifdef S_IFIFO
    case S_IFIFO:
       break;
#endif
  }
 return(how);
}

static void printent(REMOTE *r, const char *pref)
{
 ENTRY *e;
 time_t ttmp;

 e = r->dirlist;
 switch (e->mode & S_IFMT)
  { default:
       printf("%sunknown mode %o",pref,e->mode);
       break;
    case S_IFDIR:
       printf("%s%s",pref,modestring(e->mode));
       if (! flag_noown) printf(" (owner %d/%d)",e->uid,e->gid);
       break;
    case S_IFCHR:
    case S_IFBLK:
       printf("%s%s (",pref,modestring(e->mode));
       if (! flag_noown) printf("owner %d/%d, ",e->uid,e->gid);
       if (flag_links && (e->nlinks != 1)) printf("%d links, ",e->nlinks);
       printf("device (%d,%d))",(int)major(e->rdev),(int)minor(e->rdev));
       break;
    case S_IFREG:
       printf("%s%s (",pref,modestring(e->mode));
       if (! flag_noown) printf("owner %d/%d, ",e->uid,e->gid);
       if (flag_links && (e->nlinks != 1)) printf("%d links, ",e->nlinks);
       printf("size %lu)",e->size);
       break;
    case S_IFLNK:
       printf("%s%s (",pref,modestring(e->mode));
       if (! flag_noown) printf("owner %d/%d, ",e->uid,e->gid);
       printf("to %s)",e->linkto);
       break;
    case S_IFSOCK:
       printf("%s%s",pref,modestring(e->mode));
       if (! flag_noown) printf(" (owner %d/%d)",e->uid,e->gid);
       break;
#ifdef S_IFIFO
    case S_IFIFO:
       printf("%s%s",pref,modestring(e->mode));
       if (! flag_noown) printf(" (owner %d/%d)",e->uid,e->gid);
       break;
#endif
  }
 if (flag_mtimes)
  { ttmp = e->mtime;
    printf(" mtime %.24s",ctime(&ttmp));
  }
 printf(" in %s\n",r->fullarg);
}

static void union_entlist(ENTRY **listp, ENTRY *with)
{
 ENTRY *list;
 ENTRY_TCONC new;

 etc_init(&new);
 list = *listp;
 while (list && with)
  { int c;
    c = strcmp(list->name,with->name);
    if (c < 0)
     { etc_add(&new,list);
       list = list->link;
     }
    else if (c > 0)
     { etc_add(&new,copyentry(with));
       with = with->link;
     }
    else
     { etc_add(&new,list);
       list = list->link;
       with = with->link;
     }
  }
 while (with)
  { etc_add(&new,copyentry(with));
    with = with->link;
  }
 etc_append(&new,list);
 etc_end(&new);
 *listp = new.list;
}

static MVLIST *sort_mvlist(MVLIST *list, int (*fn)(MVLIST *, MVLIST *))
{
 MVLIST *l1;
 MVLIST *l2;
 MVLIST **lpp1;
 MVLIST **lpp2;
 MVLIST *l;

 if (!list || !list->link) return(list);
 lpp1 = &l1;
 lpp2 = &l2;
 while (list)
  { l = list;
    list = l->link;
    *lpp1 = l;
    lpp1 = lpp2;
    lpp2 = &l->link;
  }
 *lpp1 = 0;
 *lpp2 = 0;
 lpp1 = &list;
 while (l1 || l2)
  { if (!l2 || (l1 && (*fn)(l1,l2) < 0))
     { l = l1;
       l1 = l->link;
     }
    else
     { l = l2;
       l2 = l->link;
     }
    *lpp1 = l;
    lpp1 = &l->link;
  }
 *lpp1 = 0;
 return(list);
}

static int mvlist_sort_v_up(MVLIST *a, MVLIST *b)
{
 if (a->v < b->v) return(-1);
 if (a->v > b->v) return(1);
 return(0);
}

static void master_dir(const char *addl)
{
 REMOTE *r;
 MASK saveact;
 const char *last;
 REMOTE *nextr;
 MASK next;
 MASK dirs;
 ENTRY *e;
 ENTRY_TCONC recurse;
 int maxlen;
 int needprint;
 RSTACK rs;

 if (MASK_ISZERO(active) || MASK_ONEBIT(active)) return;
 saveact = active;
 for (EACH_REM(r))
  { if (MASK_TST(active,r->bit))
     { get_dirlist(r,addl);
       r->dirlist = entsort(r->dirlist);
     }
  }
 e = 0;
 for (EACH_REM(r)) union_entlist(&e,r->dirlist);
 rs.list = e;
 rs.cur = rs.list;
 rs.link = rstack;
 rstack = &rs;
 last = "";
 maxlen = 0;
 etc_init(&recurse);
 if (flag_update)
  { MASK mustrm;
    MASK mustinstall;
    MASK musttrunc;
    MASK mustchmod;
    MASK mustchown;
    MASK mustprint;
    MASK mustutimes;
    MASK ssmaller;
    MASK sequal;
    MASK slarger;
    MASK cdiffs;
    MVLIST *instlist;
    static void add_instlist(MASK m, unsigned long int s)
     { MVLIST *mvl;
       if (! MASK_ISZERO(m))
	{ mvl = NEW(MVLIST);
	  mvl->m = m;
	  mvl->v = s & ~511UL;
	  mvl->link = instlist;
	  instlist = mvl;
	}
     }
    if (! MASK_TST(active,umaster->bit)) panic("-u master not active in master_dir");
    while (1)
     { nextr = 0;
       for (EACH_REM(r)) if (MASK_TST(active,r->bit) && r->dirlist && ((nextr == 0) || (strcmp(r->dirlist->name,nextr->dirlist->name) < 0))) nextr = r;
       if (nextr == 0) break;
       while (rs.cur && (strcmp(rs.cur->name,nextr->dirlist->name) < 0)) rs.cur = rs.cur->link;
       MASK_ZERO(next);
       for (EACH_REM(r)) if (MASK_TST(active,r->bit) && r->dirlist && !strcmp(r->dirlist->name,nextr->dirlist->name)) MASK_SET(next,r->bit);
       newfile_init(addl,nextr->dirlist->name);
       if (! cur_pruned)
	{ MASK_ZERO(mustprint);
	  if (MASK_TST(next,umaster->bit))
	   { MASK_ZERO(mustrm);
	     MASK_ZERO(mustchmod);
	     MASK_ZERO(mustchown);
	     MASK_ZERO(mustutimes);
	     MASK_ZERO(ssmaller);
	     MASK_ZERO(sequal);
	     MASK_ZERO(slarger);
	     instlist = 0;
	     umaster->dirlist->mode = (umaster->dirlist->mode & perm_and) | perm_or;
	     for (EACH_REM(r))
	      { if (r == umaster) continue;
		if (MASK_TST(next,r->bit))
		 { int how;
		   how = entdiff(r->dirlist,umaster->dirlist);
		   if (how & ENTDIFF_TYPE)
		    { MASK_SET(mustrm,r->bit);
		    }
		   else
		    { if (how & ENTDIFF_MODE)
		       { MASK_SET(mustchmod,r->bit);
		       }
		      if (how & (ENTDIFF_UID|ENTDIFF_GID))
		       { MASK_SET(mustchown,r->bit);
		       }
		      if (how & (ENTDIFF_RDEV|ENTDIFF_LINK))
		       { MASK_SET(mustrm,r->bit);
		       }
		      if (how & ENTDIFF_SIZE)
		       { if (umaster->dirlist->size > r->dirlist->size) MASK_SET(ssmaller,r->bit); else if (umaster->dirlist->size < r->dirlist->size) MASK_SET(slarger,r->bit);
		       }
		      else
		       { MASK_SET(sequal,r->bit);
		       }
		      if (how & ENTDIFF_TIME)
		       { MASK_SET(mustutimes,r->bit);
		       }
		      if (how & ENTDIFF_NLNK)
		       { MASK_SET(mustprint,r->bit);
		       }
		    }
		 }
	      }
	     if ((umaster->dirlist->mode & S_IFMT) == S_IFREG)
	      { MVLIST *mvl;
		unsigned long int howfar;
		MASK tocheck;
		unsigned long int s;
		MASK cmp;
		MASK diffs;
		MASK eqsize;
		MASK_ZERO(cdiffs);
		mustinstall = mask_or(mustrm,mask_andnot(active,next));
		tocheck = ssmaller;
		howfar = 0;
		while (! MASK_ISZERO(tocheck))
		 { s = ~0UL;
		   for (EACH_REM(r)) if (MASK_TST(tocheck,r->bit) && (r->dirlist->size < s)) s = r->dirlist->size;
		   MASK_ZERO(cmp);
		   MASK_ZERO(eqsize);
		   for (EACH_REM(r)) if (MASK_TST(tocheck,r->bit))
		    { if (r->dirlist->size >= s) MASK_SET(cmp,r->bit);
		      if (r->dirlist->size == s) MASK_SET(eqsize,r->bit);
		    }
		   if (s == howfar)
		    { if (s) panic("master_dir without advancement");
		      MASK_ZERO(diffs);
		    }
		   else
		    { diffs = cmp;
		      check_reg_diff(umaster,addl,&diffs,howfar,s);
		    }
		   add_instlist(diffs,howfar);
		   add_instlist(mask_andnot(eqsize,diffs),s);
		   for (EACH_REM(r)) if (MASK_TST(tocheck,r->bit) && (r->dirlist->size >= s)) MASK_SET(cmp,r->bit);
		   howfar = s;
		   mask_setandnot(&tocheck,cmp);
		 }
		diffs = mask_or(slarger,sequal);
		check_reg_diff(umaster,addl,&diffs,0,umaster->dirlist->size);
		add_instlist(mask_or(diffs,mustinstall),0);
		cdiffs = mask_and(diffs,sequal);
		musttrunc = mask_andnot(slarger,diffs);
		mask_setor(&mustprint,musttrunc);
		instlist = sort_mvlist(instlist,mvlist_sort_v_up);
		mvl = instlist;
		while (mvl)
		 { if (mvl->link && (mvl->link->v == mvl->v))
		    { MVLIST *t;
		      t = mvl->link;
		      mask_setor(&mvl->m,t->m);
		      mvl->link = t->link;
		      free(t);
		    }
		   else
		    { mask_setor(&mustinstall,mvl->m);
		      mvl = mvl->link;
		    }
		 }
		if (flag_debug)
		 { fprintf(stderr,"instlist:\n");
		   for (mvl=instlist;mvl;mvl=mvl->link) fprintf(stderr,"\tmask %08lx v %lu\n",mvl->m.bits[0],mvl->v);
		 }
	      }
	     else
	      { mustinstall = mask_or(mask_andnot(active,next),mustrm);
		mask_setor(&mustutimes,mustinstall);
		MASK_ZERO(musttrunc);
	      }
	   }
	  else
	   { if (flag_nodelete)
	      { MASK_ZERO(mustrm);
	      }
	     else
	      { mustrm = next;
	      }
	     MASK_ZERO(mustinstall);
	     MASK_ZERO(mustchmod);
	     MASK_ZERO(mustchown);
	     MASK_ZERO(mustutimes);
	     MASK_ZERO(musttrunc);
	   }
	  mask_setandnot(&mustchmod,mustrm);
	  mask_setandnot(&mustchown,mustrm);
	  mask_setor(&mustprint,mask_or(mask_or(mustrm,mustinstall),mask_or(mustchmod,mustchown)));
	  if (flag_mtimes) mask_setor(&mustprint,mustutimes);
	  if (!MASK_ISZERO(mustprint))
	   { showhdg();
	     if (MASK_TST(next,umaster->bit))
	      { printent(umaster,"  * ");
	      }
	     else
	      { printf("  * nonexistent in %s\n",umaster->fullarg);
	      }
	     for (EACH_REM(r))
	      { if (r == umaster) continue;
		if (MASK_TST(mustprint,r->bit))
		 { if (MASK_TST(next,r->bit))
		    { printent(r,"    ");
		    }
		   else
		    { printf("    nonexistent in %s\n",r->fullarg);
		    }
		 }
	      }
	     if (MASK_TST(next,umaster->bit) && ((umaster->dirlist->mode & S_IFMT) == S_IFREG) && !MASK_ISZERO(cdiffs))
	      { printf("    plain file %s differing from master copy:\n",flag_md5?"checksums":"contents");
		for (EACH_REM(r)) if (MASK_TST(cdiffs,r->bit)) printf("\t%s\n",r->fullarg);
	      }
	   }
	  MASK_ZERO(dirs);
	  if (MASK_TST(next,umaster->bit))
	   { ENTRY *e;
	     e = umaster->dirlist;
	     for (EACH_REM(r)) if (MASK_TST(mustrm,r->bit)) printf("      removal from %s %s\n",r->fullarg,rmrm(r,addl,e->name)?"failed":"succeeded");
	     if (! MASK_ISZERO(mustinstall))
	      { if ((umaster->dirlist->mode & S_IFMT) == S_IFREG)
		 { MVLIST *mvl;
		   MASK m;
		   MASK_ZERO(m);
		   for (mvl=instlist;mvl;mvl=mvl->link)
		    { unsigned long int nextsize;
		      nextsize = mvl->link ? mvl->link->v : e->size;
		      mask_setor(&m,mvl->m);
		      installreg(m,addl,e->name,0,mvl->v,nextsize);
		    }
		   if (flag_mtimes) rmutimes(m,addl,e->name,0);
		 }
		else
		 { installit(mustinstall,addl,0);
		   if (flag_mtimes) rmutimes(mustinstall,addl,e->name,0);
		 }
	      }
	     if (! MASK_ISZERO(mustchown)) rmchown(mustchown,addl,e->name,0);
	     if (! MASK_ISZERO(musttrunc)) rmtrunc(musttrunc,addl,e->name,0);
	     if (! MASK_ISZERO(mustchmod)) rmchmod(mustchmod,addl,e->name,0);
	     if (flag_mtimes && !MASK_ISZERO(mustutimes)) rmutimes(mustutimes,addl,e->name,0);
	     if ((umaster->dirlist->mode & S_IFMT) == S_IFDIR) dirs = next;
	   }
	  else
	   { for (EACH_REM(r)) if (MASK_TST(mustrm,r->bit)) printf("      removal from %s %s\n",r->fullarg,rmrm(r,addl,nextr->dirlist->name)?"failed":"succeeded");
	   }
	  endfile_flush();
	  if (! MASK_ISZERO(dirs))
	   { int l;
	     for (EACH_REM(r)) if (MASK_TST(dirs,r->bit)) break;
	     e = r->dirlist;
	     e->present = dirs;
	     etc_add(&recurse,e);
	     l = strlen(e->name);
	     if (l > maxlen) maxlen = l;
	     r->dirlist = r->dirlist->link;
	     MASK_CLR(next,r->bit);
	   }
	}
       for (EACH_REM(r))
	{ if (MASK_TST(next,r->bit))
	   { e = r->dirlist;
	     r->dirlist = e->link;
	     freeentry(e);
	   }
	}
     }
  }
 else
  { MASK todo;
    MASK these;
    MASK ignnx;
    while (1)
     { nextr = 0;
       for (EACH_REM(r)) if (MASK_TST(active,r->bit) && r->dirlist && ((nextr == 0) || (strcmp(r->dirlist->name,nextr->dirlist->name) < 0))) nextr = r;
       if (nextr == 0) break;
       while (rs.cur && (strcmp(rs.cur->name,nextr->dirlist->name) < 0)) rs.cur = rs.cur->link;
       MASK_ZERO(next);
       MASK_ZERO(ignnx);
       for (EACH_REM(r))
	{ if (MASK_TST(active,r->bit))
	   { if (r->dirlist && !strcmp(r->dirlist->name,nextr->dirlist->name)) MASK_SET(next,r->bit);
	     if (r->ignnx) MASK_SET(ignnx,r->bit);
	   }
	}
       newfile_init(addl,nextr->dirlist->name);
       if (! cur_pruned)
	{ MASK_ZERO(dirs);
	  for (EACH_REM(r)) if (MASK_TST(next,r->bit) && ((r->dirlist->mode & S_IFMT) == S_IFDIR)) MASK_SET(dirs,r->bit);
	  for (EACH_REM(r)) if (MASK_TST(next,r->bit) && entdiff(r->dirlist,nextr->dirlist)) break;
	  cur_nx = mask_andnot(active,next);
	  if (r || !MASK_ISZERO(mask_andnot(cur_nx,ignnx)))
	   { showhdg();
	     todo = next;
	     while (! MASK_ISZERO(todo))
	      { for (EACH_REM(r)) if (MASK_TST(todo,r->bit)) break;
		e = r->dirlist;
		MASK_ZERO(these);
		for (EACH_REM(r))
		 { if (MASK_TST(todo,r->bit) && (entdiff(e,r->dirlist) == 0))
		    { printent(r,"    ");
		      MASK_SET(these,r->bit);
		    }
		 }
		mask_setandnot(&todo,these);
	      }
	     todo = next;
	     for (EACH_REM(r)) if (MASK_TST(todo,r->bit) && ((r->dirlist->mode & S_IFMT) != S_IFREG)) MASK_CLR(todo,r->bit);
	     needprint = 0;
	     while (! MASK_ISZERO(todo))
	      { REMOTE *r2;
		for (EACH_REM(r2)) if (MASK_TST(todo,r2->bit)) break;
		MASK_ZERO(these);
		for (EACH_REM(r)) if (MASK_TST(todo,r->bit) && (r->dirlist->size == r2->dirlist->size)) MASK_SET(these,r->bit);
		if (! MASK_EQ(these,todo)) needprint = 1;
		compare_reg(these,r2->dirlist->size,needprint);
		mask_setandnot(&todo,these);
	      }
	   }
	  else
	   { e = nextr->dirlist;
	     if ((e->mode & S_IFMT) == S_IFREG)
	      { compare_reg(next,e->size,0);
	      }
	   }
	  endfile_flush();
	  if (! MASK_ISZERO(dirs))
	   { int l;
	     for (EACH_REM(r)) if (MASK_TST(dirs,r->bit)) break;
	     e = r->dirlist;
	     e->present = dirs;
	     etc_add(&recurse,e);
	     l = strlen(e->name);
	     if (l > maxlen) maxlen = l;
	     r->dirlist = r->dirlist->link;
	     MASK_CLR(next,r->bit);
	   }
	}
       for (EACH_REM(r))
	{ if (MASK_TST(next,r->bit))
	   { e = r->dirlist;
	     r->dirlist = e->link;
	     freeentry(e);
	   }
	}
     }
  }
 e = rs.list;
 rs.list = 0;
 rs.cur = 0;
 while (e)
  { ENTRY *e2;
    e2 = e->link;
    freeentry(e);
    e = e2;
  }
 etc_end(&recurse);
 if (recurse.list)
  { int oldlen;
    char *newname;
    oldlen = strlen(addl);
    newname = malloc(oldlen+1+maxlen+1);
    bcopy(addl,newname,oldlen);
    newname[oldlen] = '/';
    rs.list = recurse.list;
    for (e=recurse.list;e;e=e->link)
     { strcpy(newname+oldlen+1,e->name);
       active = e->present;
       rs.cur = e;
       master_dir(newname);
     }
    rs.list = 0;
    while (recurse.list)
     { e = recurse.list->link;
       freeentry(recurse.list);
       recurse.list = e;
     }
    free(newname);
  }
 rstack = rs.link;
 active = saveact;
}

static FILE *siginfo_outf;

#define f siginfo_outf
static void print_rstack(RSTACK *rs, int verbose)
{
 ENTRY *e;
 int n;

 if (! rs) return;
 print_rstack(rs->link,verbose);
 if (! rs->list) return;
 if (! verbose)
  { fprintf(f,"%s",rs->cur->name);
    if ((rs->cur->mode & S_IFMT) == S_IFDIR) fprintf(f,"/");
    return;
  }
 for (n=10,e=rs->list;e&&(n>0);n--,e=e->link) ;
 if (e)
  { if (! rs->cur)
     { while (e->link) e = e->link;
       fprintf(f,"%s ... %s",rs->list->name,e->name);
     }
    else
     { e = rs->cur;
       if (e == rs->list)
	{ fprintf(f,"[ %s ]",e->name);
	}
       else if (e == rs->list->link)
	{ fprintf(f,"%s [ %s ]",rs->list->name,e->name);
	}
       else if (e == rs->list->link->link)
	{ fprintf(f,"%s %s [ %s ]",rs->list->name,rs->list->link->name,e->name);
	}
       else if (e == rs->list->link->link->link)
	{ fprintf(f,"%s %s %s [ %s ]",rs->list->name,rs->list->link->name,rs->list->link->link->name,e->name);
	}
       else if (e == rs->list->link->link->link->link)
	{ fprintf(f,"%s %s %s %s [ %s ]",rs->list->name,rs->list->link->name,rs->list->link->link->name,rs->list->link->link->link->name,e->name);
	}
       else
	{ ENTRY *e1;
	  ENTRY *e2;
	  ENTRY *e3;
	  fprintf(f,"%s ... ",rs->list->name);
	  e1 = 0;
	  e2 = 0;
	  e3 = 0;
	  for (e=rs->list;e&&(e!=rs->cur);e=e->link)
	   { e1 = e2;
	     e2 = e3;
	     e3 = e;
	   }
	  if (e)
	   { fprintf(f,"%s %s %s [ %s ]",e1->name,e2->name,e3->name,e->name);
	   }
	  else
	   { fprintf(f,"%s [?can't find current entry]",e3->name);
	   }
	}
       if (e)
	{ n = 0;
	  for (e=e->link;e;n++,e=e->link)
	   { if (!e->link || (n < 3)) fprintf(f," %s",e->name);
	     if ((n == 2) && e->link && e->link->link) fprintf(f," ...");
	   }
	}
     }
  }
 else
  { for (e=rs->list;e;e=e->link)
     { if (e == rs->cur)
	{ fprintf(f,"[ %s ]%s",e->name,e->link?" ":"");
	}
       else
	{ fprintf(f,"%s%s",e->name,e->link?" ":"");
	}
     }
  }
 fprintf(f,"\n");
}
#undef f

#ifdef SIGINFO

static void siginfo(int sig __attribute__((__unused__)))
{
 print_rstack(rstack,1);
 fprintf(siginfo_outf,"\n");
 fflush(siginfo_outf);
}

#endif

static void sigalrm(int sig __attribute__((__unused__)))
{
 print_rstack(rstack,0);
 fprintf(siginfo_outf,"\n");
 fflush(siginfo_outf);
}

static void runmaster(void)
{
 REMOTE *r;
 int bit;

 bit = 0;
 MASK_ZERO(active);
 for (EACH_REM(r))
  { r->bit = bit;
    MASK_SET(active,bit);
    bit ++;
  }
 if (bit > MAXREMOTES)
  { fprintf(stderr,"%s: too many directories specified\n",__progname);
    exit(1);
  }
 if (flag_update) for (EACH_REM(r)) umaster = r;
 rstack = 0;
 siginfo_outf = fopen("/dev/tty","w");
 if (siginfo_outf) signal(SIGALRM,sigalrm);
 if (tick_seconds > 0)
  { struct itimerval itv;
    itv.it_interval.tv_sec = tick_seconds;
    itv.it_interval.tv_usec = 0;
    itv.it_value = itv.it_interval;
    setitimer(ITIMER_REAL,&itv,0);
  }
#ifdef SIGINFO
 if (siginfo_outf) signal(SIGINFO,siginfo);
#endif
 master_dir(".");
 for (EACH_REM(r)) fflush(r->wf);
}

static void master(void)
{
 if (remotes == 0)
  { fprintf(stderr,"%s: no directories specified\n",__progname);
    exit(1);
  }
 if (remotes->link == 0)
  { fprintf(stderr,"%s: only one directory specified\n",__progname);
    exit(1);
  }
 runremotes();
 startup_remotes();
 runmaster();
}

static void add_prune(char *path, int type)
{
 PRUNE *p;

 if ((path[0] == '.') && (path[1] == '/'))
  { path += 2;
    while (path[0] == '/') path ++;
  }
 p = NEW(PRUNE);
 p->type = type;
 p->path = path;
 p->link = prunelist;
 prunelist = p;
}

static void getdq(const char *s, char **epp, unsigned long int *ap)
{
 char *ep;
 unsigned char o[4];
 long int v;
 int i;

 for (i=0;i<4;i++)
  { if (i > 0)
     { if (*s != '.')
	{ fprintf(stderr,"%s: invalid dotted quad\n",__progname);
	  exit(1);
	}
       s ++;
     }
    v = strtol(s,&ep,0);
    if ((s == ep) || (v < 0) || (v > 255))
     { fprintf(stderr,"%s: missing/invalid octet in dotted quad\n",__progname);
       exit(1);
     }
    o[i] = v;
    s = ep;
  }
 *ap = (o[0]*0x01000000UL) |
       (o[1]*0x00010000UL) |
       (o[2]*0x00000100UL) |
       (o[3]             );
 *epp = ep;
}

static void setup_accept(const char *s)
{
 char *ep;
 long int v;

 v = strtol(s,&ep,0);
 if ((ep == s) || (v < 0) || (v > 65535))
  { fprintf(stderr,"%s: missing/invalid port number\n",__progname);
    exit(1);
  }
 conn_sin.sin_port = htons(v);
 if (*ep)
  { if (*ep != ':')
     { fprintf(stderr,"%s: trash after port number\n",__progname);
       exit(1);
     }
    getdq(ep+1,&ep,&peer_addr);
    if (*ep == '/')
     { s = ep + 1;
       v = strtol(s,&ep,0);
       if (! *ep)
	{ if ((s == ep) || (v < 0) || (v > 32))
	   { fprintf(stderr,"%s: missing/invalid netmask\n",__progname);
	     exit(1);
	   }
	  switch (v)
	   { case 0: peer_mask = 0; break;
	     case 32: peer_mask = 0xffffffff; break;
	     default: peer_mask = ((~0UL)<<v)<<(32-v); break;
	   }
	}
       else
	{ getdq(s,&ep,&peer_mask);
	  if (*ep)
	   { fprintf(stderr,"%s: trash after netmask\n",__progname);
	     exit(1);
	   }
	}
       peer_addr &= peer_mask;
     }
    else if (! *ep)
     { peer_mask = ~0UL;
     }
    else
     { fprintf(stderr,"%s: trash after peer address\n",__progname);
       exit(1);
     }
  }
 else
  { peer_addr = 0;
    peer_mask = 0;
  }
 flag_accept = 1;
}

int main(int, char **);
int main(int ac, char **av)
{
 remotes = 0;
 defprogram = av[0];
 perm_mask = 07777;
 tick_seconds = 0;
 signal(SIGALRM,SIG_IGN);
#define SKIP(n) do{ac-=(n);av+=(n);}while(0)
 while (ac > 1)
  { SKIP(1);
    if (!strcmp(av[0],"-debug") || !strcmp(av[0],"-D"))
     { flag_debug = 1;
       continue;
     }
    if (!strcmp(av[0],"-follow") || !strcmp(av[0],"-h"))
     { flag_follow = 1;
       continue;
     }
    if (!strcmp(av[0],"-sgid-dirs") || !strcmp(av[0],"-g"))
     { flag_sgiddir = 1;
       continue;
     }
    if (!strcmp(av[0],"-no-owners") || !strcmp(av[0],"-o"))
     { flag_noown = 1;
       continue;
     }
    if (!strcmp(av[0],"-R"))
     { flag_remote = 1;
       continue;
     }
    if (!strcmp(av[0],"-mtimes") || !strcmp(av[0],"-t"))
     { flag_mtimes = 1;
       continue;
     }
    if (!strcmp(av[0],"-update") || !strcmp(av[0],"-u"))
     { flag_update = 1;
       continue;
     }
    if (!strcmp(av[0],"-encode") || !strcmp(av[0],"-x"))
     { flag_encode = 1;
       continue;
     }
    if (!strcmp(av[0],"-trace") || !strcmp(av[0],"-X"))
     { flag_trace = 1;
       continue;
     }
    if (!strcmp(av[0],"-readonly") || !strcmp(av[0],"-ro"))
     { flag_readonly = 1;
       continue;
     }
    if (!strcmp(av[0],"-check"))
     { flag_check = 1;
       continue;
     }
    if (!strcmp(av[0],"-no-sparse"))
     { flag_nosparse = 1;
       continue;
     }
    if (!strcmp(av[0],"-links"))
     { flag_links = 1;
       continue;
     }
    if (!strcmp(av[0],"-ignore-nonexistent") || !strcmp(av[0],"-inx"))
     { flag_nextinx = 1;
       continue;
     }
    if (!strcmp(av[0],"-md5"))
     { flag_md5 = 1;
       continue;
     }
    if (!strcmp(av[0],"-no-delete"))
     { flag_nodelete = 1;
       continue;
     }
    if (!strcmp(av[0],"-accept") && av[1])
     { setup_accept(av[1]);
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-gzip") && av[1])
     { gzip_arg = av[1];
       flag_gzip = 1;
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-prune") && av[1])
     { add_prune(av[1],PT_PATH);
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-prunewild") && av[1])
     { add_prune(av[1],PT_WILD);
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-connect") && av[1] && av[2])
     { conn_sin.sin_addr.s_addr = inet_addr(av[1]);
       conn_sin.sin_port = htons(atoi(av[2]));
       flag_connect = 1;
       SKIP(2);
       continue;
     }
    if (!strcmp(av[0],"-prunex"))
     { prunex = 1;
       continue;
     }
    if (!strcmp(av[0],"-bg"))
     { flag_bg = 1;
       continue;
     }
    if (!strcmp(av[0],"-limit"))
     { flag_limitslave = 1;
       continue;
     }
    if (!strcmp(av[0],"-rsh") && av[1])
     { currsh = av[1];
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-dir") && av[1])
     { newremote(av[1]);
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-tick") && av[1])
     { tick_seconds = atoi(av[1]);
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-mask") && av[1])
     { perm_mask = strtol(av[1],0,8) & 07777;
       SKIP(1);
       continue;
     }
    if (!strcmp(av[0],"-modemod") && av[1] && av[2])
     { perm_and = (strtol(av[1],0,8) & 07777) | (~0U << 12);
       perm_or = strtol(av[2],0,8) & 07777;
       SKIP(2);
       continue;
     }
    if (!strcmp(av[0],"-force") && av[1])
     { force_dir = av[1];
       SKIP(1);
       continue;
     }
    if (av[0][0] == '-')
     { fprintf(stderr,"%s: unrecognized flag %s\n",__progname,av[0]);
       exit(1);
     }
    newremote(av[0]);
  }
#undef SKIP
 bzero(&zeroblk[0],WORKSIZE);
 if (flag_remote)
  { if (remotes)
     { fprintf(stderr,"%s: no directories may be given when using -R\n",__progname);
       exit(1);
     }
    if (flag_accept)
     { int s;
       s = setup_connection_accept(0,conn_sin.sin_port,peer_addr,peer_mask);
       if (s != 0)
	{ dup2(s,0);
	  close(s);
	}
       dup2(0,1);
     }
    else if (flag_connect)
     { int s;
       s = setup_connection_connect(0,conn_sin.sin_addr,conn_sin.sin_port);
       if (s != 0)
	{ dup2(s,0);
	  close(s);
	}
       dup2(0,1);
     }
    maybe_background();
    slave();
  }
 else
  { if (flag_readonly)
     { fprintf(stderr,"%s: can't use -readonly except with -R\n",__progname);
       exit(1);
     }
    if (flag_connect)
     { fprintf(stderr,"%s: can't use -connect except with -R\n",__progname);
       exit(1);
     }
    if (flag_accept)
     { fprintf(stderr,"%s: can't use -accept except with -R\n",__progname);
       exit(1);
     }
    if (flag_bg)
     { fprintf(stderr,"%s: can't use -bg except with -R\n",__progname);
       exit(1);
     }
    master();
  }
 exit(0);
}
